/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.net;

import com.lf.vfslib.VFSLib;
import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.dropbox.DbxFileSystemConfigBuilder;
import com.lf.vfslib.gdrive.GDriveFileSystemConfigBuilder;
import com.lf.vfslib.s3.S3FileSystemConfigBuilder;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.provider.ftp.FtpFileSystemConfigBuilder;
import org.apache.commons.vfs2.provider.ftps.FtpsFileSystemConfigBuilder;
import org.apache.commons.vfs2.provider.sftp.SftpFileSystemConfigBuilder;

import java.lang.reflect.Array;
import java.util.Properties;
import java.util.UUID;
import java.util.Vector;


/**
 * Data holder class representing VFS (virtual file system) connections based on URLs.
 * <p>
 * Objects of this type are managed using the <code>VFSConnectionDialog</code>.
 * Subclasses may add URL specific parameters as necessary.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class VFSConnection {


    /**
     * Configuration variable name (<code>String</code>).
     */
    public final static String VAR_CONNECTION_ID = "VAR_CONNECTION_ID";
    /**
     * Configuration variable name (<code>String</code>).
     */
    public final static String VAR_CONNECTION_NAME = "VAR_CONNECTION_NAME";
    /**
     * Configuration variable name (<code>String</code>).
     */
    public final static String VAR_PROTOCOL = "VAR_PROTOCOL";
    /**
     * Configuration variable name (<code>boolean</code>).
     */
    public final static String VAR_EXPERT_MAX_WAIT = "VAR_EXPERT_MAX_WAIT";
    /**
     * Configuration variable name (<code>int</code>).
     */
    public final static String VAR_EXPERT_MAX_WAIT_SECONDS = "VAR_EXPERT_MAX_WAIT_SECONDS";
    /**
     * Configuration variable name (<code>boolean</code>).
     */
    public final static String VAR_EXPERT_KEEP_ALIVE = "VAR_EXPERT_KEEP_ALIVE";
    /**
     * Configuration variable name (<code>int</code>).
     */
    public final static String VAR_EXPERT_KEEP_ALIVE_SECONDS = "VAR_EXPERT_KEEP_ALIVE_SECONDS";
    /**
     * Configuration variable name (<code>boolean</code>).
     */
    public final static String VAR_EXPERT_KEEP_ALIVE_PING = "VAR_EXPERT_KEEP_ALIVE_PING";
    /**
     * Configuration variable name (<code>boolean</code>).
     */
    public final static String VAR_EXPERT_KEEP_ALIVE_RESOLVE = "VAR_EXPERT_KEEP_ALIVE_RESOLVE";
    /**
     * Configuration variable name (<code>boolean</code>).
     */
    public final static String VAR_EXPERT_DISPLAY_BEFORE = "VAR_EXPERT_DISPLAY_BEFORE";
    /**
     * Configuration variable name (<code>String</code>).
     */
    public final static String VAR_EXPERT_ENCODING = "VAR_EXPERT_ENCODING";

    /**
     * The ip address or the name of the host (<code>String</code>).
     */
    public final static String PARAM_HOST = "PARAM_HOST";
    /**
     * The port number for the host access (<code>int</code>).
     */
    public final static String PARAM_PORT = "PARAM_PORT";
    /**
     * The (absolute) path on the host (<code>String</code>).
     */
    public final static String PARAM_PATH = "PARAM_PATH";
    /**
     * The name of the user (<code>String</code>).
     */
    public final static String PARAM_USER = "PARAM_USER";
    /**
     * The password for the user (<code>String</code>).
     */
    public final static String PARAM_USER_PASSWORD = "PARAM_USER_PWD";
    /**
     * Save the password for the user (<code>boolean</code>)?
     */
    public final static String PARAM_SAVE_PASSWORD = "PARAM_SAVE_PWD";

    /**
     * The file system manager.
     */
    protected VFSFileSystemManager vfsManager = null;
    /**
     * The settings (driver parameters, expert settings).
     */
    protected Properties props = null;
    /**
     * Scheme specific options like <code>FtpFileSystemConfigBuilder.getInstance().setPassiveMode()</code>.
     */
    protected FileSystemOptions options = null;


    /**
     * Constructor method for i18n purposes only.
     *
     * @throws InstantiationException Error indication
     */
    public VFSConnection() throws InstantiationException {

        if (!java.beans.Beans.isDesignTime()) {
            throw new InstantiationException("Do not use this constructor!");
        }
    }

    /**
     * Constructor method setting default values.
     *
     * @param vfsmanager The file system manager
     * @param id         The unique connection identifier
     * @throws NullPointerException If parameter is <code>null</code>
     */
    public VFSConnection(VFSFileSystemManager vfsmanager, String id) {

        if (vfsmanager == null) throw new NullPointerException();
        if (id == null) throw new NullPointerException();

        this.vfsManager = vfsmanager;
        this.props = new Properties();
        this.options = new FileSystemOptions();  // Set default options if the protocol/scheme is known

        // Should be the initial values of the VFSConnectionDialog
        this.props.setProperty(VAR_CONNECTION_ID, id);
        this.props.setProperty(VAR_CONNECTION_NAME, "");
        this.props.setProperty(VAR_PROTOCOL, "");
        this.props.setProperty(VAR_EXPERT_MAX_WAIT, Boolean.FALSE.toString());
        this.props.setProperty(VAR_EXPERT_MAX_WAIT_SECONDS, "30");
        this.props.setProperty(VAR_EXPERT_KEEP_ALIVE, Boolean.FALSE.toString());
        this.props.setProperty(VAR_EXPERT_KEEP_ALIVE_SECONDS, "60");
        this.props.setProperty(VAR_EXPERT_KEEP_ALIVE_PING, Boolean.TRUE.toString());
        this.props.setProperty(VAR_EXPERT_KEEP_ALIVE_RESOLVE, Boolean.FALSE.toString());
        this.props.setProperty(VAR_EXPERT_DISPLAY_BEFORE, Boolean.FALSE.toString());
        this.props.setProperty(VAR_EXPERT_ENCODING, "");

        this.props.setProperty(PARAM_HOST, "");
        this.props.setProperty(PARAM_PORT, "-1");
        this.props.setProperty(PARAM_PATH, "");
        this.props.setProperty(PARAM_USER, "");
        this.props.setProperty(PARAM_USER_PASSWORD, "");
        this.props.setProperty(PARAM_SAVE_PASSWORD, Boolean.FALSE.toString());
    }

    /**
     * Constructor method setting the container.
     *
     * @param vfsmanager The file system manager
     * @param id         The unique connection identifier
     * @param props      The container, empty is OK
     * @param options    Scheme specific options like <code>FtpFileSystemConfigBuilder.getInstance().setPassiveMode()</code>
     * @throws NullPointerException If parameter is <code>null</code>
     */
    public VFSConnection(VFSFileSystemManager vfsmanager, String id, Properties props, FileSystemOptions options) {

        if (vfsmanager == null) throw new NullPointerException();
        if (id == null) throw new NullPointerException();
        if (props == null) throw new NullPointerException();
        if (options == null) throw new NullPointerException();

        this.vfsManager = vfsmanager;
        this.props = props;
        this.props.setProperty(VAR_CONNECTION_ID, id);
        this.options = options;
    }

    /**
     * Provides the file system manager.
     *
     * @return File system manager
     */
    public VFSFileSystemManager getFileSystemManager() {
        return this.vfsManager;
    }

    /**
     * The file system specific options for this connection.
     *
     * @return The options, may be <code>null</code>
     */
    public FileSystemOptions getFileSystemOptions() {
        return this.options;
    }

    /**
     * The file system specific options for this connection.
     *
     * @param options The options, may be empty
     */
    public void setFileSystemOptions(FileSystemOptions options) {
        this.options = options;
    }

    /**
     * Any file system specific options for this connection available (internal cache not empty)?
     *
     * @return Options available?
     */
    public boolean hasFileSystemOptions() {
        return VFSUtils.hasFileSystemOptions(this.options);
    }

    /**
     * Convenience method to get the info if the underlying VFS scheme is available.
     * <p>
     * Availability here means if the VFS driver is available on the classpath.
     *
     * @return Connection available?
     */
    public boolean isAvailable() {

        try {
            String[] schemes = this.vfsManager.getSchemes();
            String protocol = this.getProtocol(null);
            return elementOf(protocol, schemes);
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Provides the unique connection identifier.
     *
     * @param fallback The default value
     * @return Connection name, may be <code>null</code>
     */
    public String getID(String fallback) {
        return this.props.getProperty(VAR_CONNECTION_ID, fallback);
    }

    /**
     * Provides the connection name.
     *
     * @param fallback The default value
     * @return Connection name, may be <code>null</code>
     */
    public String getName(String fallback) {
        return this.props.getProperty(VAR_CONNECTION_NAME, fallback);
    }

    /**
     * Sets the connection name.
     *
     * @param value The connection name, removed if <code>null</code>
     */
    public void setName(String value) {

        if (value != null) this.props.setProperty(VAR_CONNECTION_NAME, value);
        else this.props.remove(VAR_CONNECTION_NAME);
    }

    /**
     * Provides the type of protocol (aka scheme like "sftp").
     *
     * @param fallback The default value
     * @return Protocol, may be <code>null</code>
     */
    public String getProtocol(String fallback) {
        return this.props.getProperty(VAR_PROTOCOL, fallback);
    }

    /**
     * Sets the type of protocol (aka scheme like "sftp").
     *
     * @param value The protocol, removed if <code>null</code>
     */
    public void setProtocol(String value) {

        if (value != null) this.props.setProperty(VAR_PROTOCOL, value);
        else this.props.remove(VAR_PROTOCOL);
    }

    /**
     * Provides the expert setting if to wait for connect.
     *
     * @param fallback The default value
     * @return Wait for connect?
     */
    public boolean getMaxWait(boolean fallback) {
        return Boolean.parseBoolean(this.props.getProperty(VAR_EXPERT_MAX_WAIT, String.valueOf(fallback)));
    }

    /**
     * Provides the maximum number of seconds to wait for connect.
     *
     * @param fallback The default value
     * @return Seconds
     */
    public int getMaxWaitSeconds(int fallback) {
        return Integer.parseInt(this.props.getProperty(VAR_EXPERT_MAX_WAIT_SECONDS, String.valueOf(fallback)));
    }

    /**
     * Provides the expert setting if connection is kept alive automatically.
     *
     * @param fallback The default value
     * @return Keep alive?
     */
    public boolean getKeepAlive(boolean fallback) {
        return Boolean.parseBoolean(this.props.getProperty(VAR_EXPERT_KEEP_ALIVE, String.valueOf(fallback)));
    }

    /**
     * Provides the keep alive interval.
     *
     * @param fallback The default value
     * @return Seconds
     */
    public int getKeepAliveSeconds(int fallback) {
        return Integer.parseInt(this.props.getProperty(VAR_EXPERT_KEEP_ALIVE_SECONDS, String.valueOf(fallback)));
    }

    /**
     * Provides the expert setting if connection is kept alive with ping.
     *
     * @param fallback The default value
     * @return Keep alive via ping?
     */
    public boolean getKeepAlivePing(boolean fallback) {
        return Boolean.parseBoolean(this.props.getProperty(VAR_EXPERT_KEEP_ALIVE_PING, String.valueOf(fallback)));
    }

    /**
     * Provides the expert setting if connection is kept alive with entry resolves.
     *
     * @param fallback The default value
     * @return Keep alive via resolves?
     */
    public boolean getKeepAliveResolve(boolean fallback) {
        return Boolean.parseBoolean(this.props.getProperty(VAR_EXPERT_KEEP_ALIVE_RESOLVE, String.valueOf(fallback)));
    }

    /**
     * Provides the expert setting if settings shall be displayed before connection is established.
     *
     * @param fallback The default value
     * @return Display settings?
     */
    public boolean getDisplayBefore(boolean fallback) {
        return Boolean.parseBoolean(this.props.getProperty(VAR_EXPERT_DISPLAY_BEFORE, String.valueOf(fallback)));
    }

    /**
     * Provides the expert setting which file system encoding shall be used.
     *
     * @param fallback The default value
     * @return Encoding
     */
    public String getEncoding(String fallback) {
        return this.props.getProperty(VAR_EXPERT_ENCODING, fallback);
    }

    /**
     * Provides the URL host part.
     *
     * @param fallback The default value
     * @return URL host, may be <code>null</code>
     */
    public String getHost(String fallback) {
        return this.props.getProperty(PARAM_HOST, fallback);
    }

    /**
     * Provides the URL port part.
     *
     * @param fallback The default value
     * @return URL port, may be <code>null</code>
     */
    public int getPort(int fallback) {
        return Integer.parseInt(this.props.getProperty(PARAM_PORT, String.valueOf(fallback)));
    }

    /**
     * Provides the URL path part.
     *
     * @param fallback The default value
     * @return URL path, may be <code>null</code>
     */
    public String getPath(String fallback) {
        return this.props.getProperty(PARAM_PATH, fallback);
    }

    /**
     * Provides the URL user part.
     *
     * @param fallback The default value
     * @return URL user, may be <code>null</code>
     */
    public String getUser(String fallback) {
        return this.props.getProperty(PARAM_USER, fallback);
    }

    /**
     * Sets the URL user part.
     *
     * @param user The user name
     */
    public void setUser(String user) {
        this.props.setProperty(PARAM_USER, user);
    }

    /**
     * Provides the URL password part.
     *
     * @param fallback The default value
     * @return URL password, may be <code>null</code>
     */
    public String getPassword(String fallback) {
        return this.props.getProperty(PARAM_USER_PASSWORD, fallback);
    }

    /**
     * Sets the URL password part.
     *
     * @param password The user password
     */
    public void setPassword(String password) {
        this.props.setProperty(PARAM_USER_PASSWORD, password);
    }

    /**
     * Provides the info if the password should be remembered.
     *
     * @param save The default value
     * @return Save password?
     */
    public boolean getSavePassword(boolean save) {
        return Boolean.parseBoolean(this.props.getProperty(PARAM_SAVE_PASSWORD, String.valueOf(save)));
    }

    /**
     * Sets the info if the password should be remembered.
     *
     * @param save Remember?
     */
    public void setSavePassword(boolean save) {
        this.props.setProperty(PARAM_SAVE_PASSWORD, String.valueOf(save));
    }

    /**
     * Provides access to the internal cache of settings (driver parameters, expert settings).
     *
     * @return The settings
     * @since 2.8
     */
    public Properties getSettings() {
        return this.props;
    }

    /**
     * Provides the info if the internal settings cache is not empty.
     *
     * @return Any settings?
     * @since 2.8
     */
    public boolean hasSettings() {
        return !this.props.isEmpty();
    }

    /**
     * Provides the next connection identifier unique for this instance.
     *
     * @param connections The container holding the IDs in use
     * @return The unique identifier
     */
    public static String getUniqueConnectionID(Vector<VFSConnection> connections) {

        int size = connections.size();
        String[] ids = new String[size];
        for (int i = 0; i < size; i++) {
            ids[i] = connections.elementAt(i).getID("");
        }

        String id = UUID.randomUUID().toString();
        while (elementOf(id, ids)) id = UUID.randomUUID().toString();
        return id;
    }

    /**
     * Creates a fresh new connection appropriate for the given VFS scheme/protocol.
     *
     * @param vfsmanager The file system manager
     * @param protocol   VFS scheme like "sftp"
     * @param id         The unique identifier
     * @return The new connection, <code>null</code> indicates error
     */
    public static VFSConnection createConnection(VFSFileSystemManager vfsmanager, String protocol, String id) {

        if (vfsmanager == null || protocol == null) return null;

        // At the moment we support only generic protocols...
        return new VFSConnection(vfsmanager, id);
    }

    /**
     * Creates a fresh new connection appropriate for the given VFS scheme/protocol.
     *
     * @param vfsmanager The file system manager
     * @param protocol   VFS scheme like "sftp"
     * @param id         The unique identifier
     * @param config     The settings to use
     * @param options    Scheme specific options like <code>FtpFileSystemConfigBuilder.getInstance().setPassiveMode()</code>
     * @return The new connection, <code>null</code> indicates error
     */
    public static VFSConnection createConnection(VFSFileSystemManager vfsmanager, String protocol, String id,
                                                 Properties config, FileSystemOptions options) {

        if (vfsmanager == null || protocol == null) return null;

        // At the moment we support only generic protocols...
        return new VFSConnection(vfsmanager, id, config, options);
    }

    /**
     * Provides the characteristic textual representation of this connection (identifier).
     *
     * @return The manager
     */
    @Override
    public String toString() {
        return this.getID(null);
    }

    /**
     * Provides the info if this connection is currently alive.
     *
     * @return Currently connected?
     */
    public boolean isConnected() {

        // Works for SFTP but not for FTP, use the root directory on the server
        // (base path may be erroneous)
        return VFSUtils.available(VFSUtils.getURL(this.vfsManager, this.getRootURI(), this));
    }

    /**
     * Convenience to get the host/IP address represented by this connection.
     *
     * @return The host name/IP address, may be <code>null</code>
     */
    public String getURIHost() {

        try {
            // Caution: Do not use URI here, since included SPACEs etc. lead to exception (OK for VFS)!
            // Caution: Do not use URL here, since "sftp" is not a supported protocol (exception)!
            return new VFSURI(this.getURI()).getHost();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Provides the info if this connection is configured properly and can therefore be used.
     * <p>
     * For cloud file systems an access token must be well-known. Although the user name is just for display here
     * we have to make sure it is unique for each scheme (do not mix up pool connections!). Also the connection
     * name must be unique for the same reasons.
     *
     * @param connections The other connections (optional)
     * @return null = connection OK, message in user's language otherwise
     */
    public String isValid(Vector<VFSConnection> connections) {

        // For cloud file systems an access token must be well-known
        String protocol = this.getProtocol("");
        if (protocol.equals(VFSLib.getSharedInstance().getSchemeDropbox()) ||
                protocol.equals(VFSLib.getSharedInstance().getSchemeGoogleDrive()) ||
                protocol.equals(VFSLib.getSharedInstance().getSchemeAmazonS3())) {

            // User name must be unique for all connections of this scheme
            boolean unique = true;
            if (connections != null && !connections.isEmpty()) unique = this.checkUniqueUser(connections);
            if (!unique) {
                return VFSLibSettings.getUserText(VFSConnection.class.getName() + "_USER_NOT_UNIQUE",
                        new String[]{"(%scheme%)"}, new String[]{protocol});
            }

            // Access tokens or secret keys are never stored as user passwords
            if (protocol.equals(VFSLib.getSharedInstance().getSchemeDropbox())) {
                String token = new DbxFileSystemConfigBuilder().getAccessToken(this.options);
                if (token == null || token.trim().isEmpty()) {
                    return VFSLibSettings.getUserText(VFSConnection.class.getName() + "_INVALID_EMPTY_TOKEN");
                }
            } else if (protocol.equals(VFSLib.getSharedInstance().getSchemeGoogleDrive())) {
                String token = new GDriveFileSystemConfigBuilder().getAccessToken(this.options);
                if (token == null || token.trim().isEmpty()) {
                    return VFSLibSettings.getUserText(VFSConnection.class.getName() + "_INVALID_EMPTY_TOKEN");
                }
            } else if (protocol.equals(VFSLib.getSharedInstance().getSchemeAmazonS3())) {
                // For S3 access/secret keys and bucket name are required
                String accesskey = new S3FileSystemConfigBuilder().getAccessKeyID(this.options);
                String secretkey = new S3FileSystemConfigBuilder().getSecretKey(this.options);
                String bucket = new S3FileSystemConfigBuilder().getBucketName(this.options);
                if (accesskey == null || accesskey.trim().isEmpty() ||
                        secretkey == null || secretkey.trim().isEmpty() ||
                        bucket == null || bucket.trim().isEmpty()) {

                    return VFSLibSettings.getUserText(VFSConnection.class.getName() + "_INVALID_S3_PARAMS");
                }
            }
        } else {
            // Connection name must be unique
            boolean unique = true;
            if (connections != null && !connections.isEmpty()) unique = this.checkUniqueName(connections);
            if (!unique) {
                return VFSLibSettings.getUserText(VFSConnection.class.getName() + "_NAME_NOT_UNIQUE");
            }
            if (this.getURI() == null)
                return VFSLibSettings.getUserText(VFSConnection.class.getName() + "_INVALID_CONNECTION");
        }
        return null;  // OK
    }

    /**
     * Provides the info if the connection host can be reached (connection test).
     *
     * @return Connection OK?
     */
    public boolean isReachable() {

        // Must not be executed by EDT since network file resolving may take a while

        FileObject target = null;

        try {
            target = this.vfsManager.resolveFile(this.getURI(), this);
        } catch (Exception | Error ignored) {
        }
        return target != null;
    }

    /**
     * Makes a copy of this object.
     *
     * @return The copy
     * @throws CloneNotSupportedException If cloning is not supported
     */
    public VFSConnection clone() throws CloneNotSupportedException {

        Properties temp = new Properties();
        temp.putAll(this.props);  // Same ID!

        FileSystemOptions options = (FileSystemOptions) this.options.clone();

        return new VFSConnection(this.vfsManager, this.getID(null), temp, options);
    }

    /**
     * Assembles the URI which can be used for VFS resolves etc.
     *
     * @return The URI, <code>null</code> indicates error
     */
    public String getURI() {
        return this.getURI(null, null);
    }

    /**
     * Assembles the URI which can be used for VFS resolves etc.
     *
     * @param username The login name, use connection credentials otherwise
     * @param password The password, use connection credentials otherwise
     * @return The URI, <code>null</code> indicates error
     */
    public String getURI(String username, String password) {
        try {
            // Example:
            // ftp://[ username [: password ]@] hostname [: port ][ absolute-path ]

            // Directly specified credentials have priority
            String userinfo = null;
            if (username == null) username = this.getUser("");
            if (password == null) password = this.getPassword("");

            if (!username.isEmpty()) {
                userinfo = this.getUser("");
                if (!password.isEmpty()) userinfo += ':' + password;
            }

            // Caution: Do not use URI here, since included SPACEs etc. lead to exception (OK for VFS)!
            // Caution: Do not use URL here, since "sftp" is not a supported protocol (exception)!
            VFSURI uri = new VFSURI(this.getProtocol(null),
                    userinfo,
                    this.getHost(null),
                    this.getPort(-1),
                    this.getPath(null),
                    null, null);  // Both included in path
            return uri.toString();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Assembles the root URI which can be used for VFS resolves etc.
     *
     * @return The root URI, <code>null</code> indicates error
     */
    public String getRootURI() {
        return this.getRootURI(null, null);
    }

    /**
     * Assembles the root URI which can be used for VFS resolves etc.
     *
     * @param username The login name, use connection credentials otherwise
     * @param password The password, use connection credentials otherwise
     * @return The root URI, <code>null</code> indicates error
     */
    public String getRootURI(String username, String password) {
        try {
            // Example:
            // ftp://[ username [: password ]@] hostname [: port ][ absolute-path ]

            // Directly specified credentials have priority
            String userinfo = null;
            if (username == null) username = this.getUser("");
            if (password == null) password = this.getPassword("");

            if (!username.isEmpty()) {
                userinfo = this.getUser("");
                if (!password.isEmpty()) userinfo += ':' + password;
            }

            // Caution: Do not use URI here, since included SPACEs etc. lead to exception (OK for VFS)!
            // Caution: Do not use URL here, since "sftp" is not a supported protocol (exception)!
            VFSURI uri = new VFSURI(this.getProtocol(null),
                    userinfo,
                    this.getHost(null),
                    this.getPort(-1),
                    null,  // No path = root
                    null, null);  // Both included in path
            return uri.toString();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Assembles the URI which can be displayed safely (no cleartext passwords).
     *
     * @return The display URI, <code>null</code> indicates error
     */
    public String getDisplayURI() {

        try {
            // Example:
            // ftp://[ username [: password ]@] hostname [: port ][ absolute-path ]

            String userinfo = null;
            if (!this.getUser("").isEmpty()) userinfo = this.getUser("");  // Omit password here

            // Caution: Do not use URI here, since included SPACEs etc. lead to exception (OK for VFS)!
            // Caution: Do not use URL here, since "sftp" is not a supported protocol (exception)!
            VFSURI uri = new VFSURI(this.getProtocol(null),
                    userinfo,
                    this.getHost(null),
                    this.getPort(-1),
                    this.getPath(null),
                    null, null);  // Both included in path
            return uri.toString();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Checks if the given entry can be processed/listed by this connection instance.
     *
     * @param url         The network file/directory
     * @param compareuser Compare user info?
     * @return Connection OK?
     */
    public boolean isApplicable(String url, boolean compareuser) {

        try {
            String userinfo = null;
            if (!this.getUser("").isEmpty()) userinfo = this.getUser("");  // Omit password here

            VFSURI uri = new VFSURI(this.getProtocol(null),
                    compareuser ? userinfo : null,
                    this.getHost(null),
                    this.getPort(-1),
                    null,  // No path here!
                    null, null);
            String displayuri = uri.toString();
            // sftp://aschwolow@mycompany.com:666
            // sftp://mycompany.com:666

            // Normalize parameter
            uri = new VFSURI(VFSUtils.getDisplayURIUsernameOnly(url, null));
            uri = new VFSURI(uri.getScheme(),
                    compareuser ? uri.getUserInfo() : null,
                    uri.getHost(),
                    uri.getPort(),
                    null,  // No path here!
                    null, null);
            String paramuri = uri.toString();

            return displayuri.equals(paramuri);
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Provides the default options for a certain protocol.
     * <p>
     * Here, all scheme specific options are being preconfigured that are supported by the dialog protocol panels.
     *
     * @param scheme   The protocol/scheme like "sftp"
     * @param encoding The encoding like UTF-8 (optional)
     * @return The options, may be empty
     * @throws NullPointerException If parameter is <code>null</code>
     */
    public static FileSystemOptions getDefaultFileSystemOptions(String scheme, String encoding) {

        FileSystemOptions options = new FileSystemOptions();
        if (scheme == null) return options;  // See OTRS 2014052911000015

        // TODO: Keep options in sync with VFS dialog protocol panels.

        try {
            switch (scheme) {
                case "sftp": {
                    // SFTP does not require strict key checking here:
                    // http://nileshbansal.blogspot.com/2007/07/accessing-files-over-sftp-in-java.html
                    SftpFileSystemConfigBuilder builder = SftpFileSystemConfigBuilder.getInstance();
                    builder.setStrictHostKeyChecking(options, "no");  // Loose checking
                    builder.setUserDirIsRoot(options, false);         // "/" is root
                    builder.setIdentityProvider(options); // No private keys e.g. for AWS EC2 instance

                    // TODO
                    //if (encoding != null && !encoding.trim().isEmpty()) builder.setControlEncoding(options, encoding);
                    break;
                }
                case "ftp": {
                    FtpFileSystemConfigBuilder builder = FtpFileSystemConfigBuilder.getInstance();
                    builder.setPassiveMode(options, false);    // Active mode
                    builder.setUserDirIsRoot(options, false);  // "/" is root

                    if (encoding != null && !encoding.trim().isEmpty()) {
                        builder.setControlEncoding(options, encoding);
                    }
                    break;
                }
                case "ftps": {
                    FtpsFileSystemConfigBuilder builder = FtpsFileSystemConfigBuilder.getInstance();
                    builder.setPassiveMode(options, false);    // Active mode
                    builder.setUserDirIsRoot(options, false);  // "/" is root

                    if (encoding != null && !encoding.trim().isEmpty()) {
                        builder.setControlEncoding(options, encoding);
                    }
                    break;
                }
            }
        } catch (Throwable ignored) {
        }

        // Catch Error as well (e.g. Google libs may not be present)
        try {
            VFSLib vfslib = VFSLib.getSharedInstance();
            if (scheme.equals(vfslib.getSchemeDropbox())) {
                //DbxFileSystemConfigBuilder builder = DbxFileSystemConfigBuilder.getSharedInstance();
                // setAccessToken() is set by user
                // setAccountDisplayName()  Set by user
            } else if (scheme.equals(vfslib.getSchemeGoogleDrive())) {
                GDriveFileSystemConfigBuilder builder = GDriveFileSystemConfigBuilder.getSharedInstance();
                // setAccessToken() is set by user
                // setAccountDisplayName()  Set by user
                builder.setUseTrash(options, true);    // Do not delete permanently, allow to be restored
            } else if (scheme.equals(vfslib.getSchemeAmazonS3())) {
                //S3FileSystemConfigBuilder builder = S3FileSystemConfigBuilder.getSharedInstance();
                // setAccessKeyID() is set by user
                // setSecretKey() is set by user
                // setAccountDisplayName() is set by user
                // setRootBucketName() is set by user
            }
        } catch (Throwable ignored) {
        }
        return options;
    }

    /**
     * Checks if the user for the given connection is unique over all connections of the same scheme.
     *
     * @param connections The other connections
     * @return User name unique?
     * @throws NullPointerException If a parameter is <code>null</code>
     */
    public boolean checkUniqueUser(Vector<VFSConnection> connections) {

        if (connections == null) throw new NullPointerException();

        try {
            String username = this.getUser("");
            String protocol = this.getProtocol("");

            for (VFSConnection next : connections) {
                if (next == this) continue;
                if (next.getID("").equals(this.getID(""))) continue;  // May have been cloned (dialog)

                String protocol2 = next.getProtocol("");
                if (!protocol.equals(protocol2)) continue;  // Compare only for "dropbox", "gdrive", "s3", ...

                String username2 = next.getUser("");
                if (username.equals(username2)) return false;
            }
            return true;
        } catch (Exception e) { /*e.printStackTrace();*/ }
        return false;
    }

    /**
     * Checks if the name for the given connection is unique over all connections.
     *
     * @param connections The other connections
     * @return User name unique?
     * @throws NullPointerException If a parameter is <code>null</code>
     */
    public boolean checkUniqueName(Vector<VFSConnection> connections) {

        if (connections == null) throw new NullPointerException();

        try {
            String name = this.getName("");

            for (VFSConnection next : connections) {
                if (next == this) continue;
                if (next.getID("").equals(this.getID(""))) continue;  // May have been cloned (dialog)

                String name2 = next.getName("");
                if (name.equals(name2)) return false;
            }
            return true;
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Checks if the given object is element of the given array (equals).
     *
     * @param object The object to be checked
     * @param array  The array to check against
     * @return Is the object element of the array?
     * @throws NullPointerException If object and/or array is <code>null</code>
     */
    public static boolean elementOf(Object object, Object array) {

        Object arrayobj;

        if (array.getClass().isArray()) {

            // For the primitive types we use auto-boxing, so let's compare objects
            for (int i = 0; i < Array.getLength(array); i++) {
                arrayobj = Array.get(array, i);
                if (arrayobj == null) continue;

                // Changed to compare File and Win32Folder objects (subclass of File)
                if (arrayobj.equals(object)) {
                    return true;
                }
            }
        }
        return false;
    }
}
