/*
/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.lang;

/**
 * This class holds various methods for common use with strings.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class StringUtils {


    /**
     * Constructor method for i18n purposes only.
     *
     * @throws InstantiationException Error indication
     */
    public StringUtils() throws InstantiationException {

        if (!java.beans.Beans.isDesignTime()) {
            throw new InstantiationException("Do not use this constructor!");
        }
    }

    /**
     * Workaround for standard regex replacement method if string contains unallowed chars.
     * <p>
     * For instance the <code>String.replaceAll()</code> method throws an exception if the string
     * to be replaced contains a dollar character ($) since this is interpreted as group. Make
     * sure that no %DOLLAR% substring is included since this is used ad intermediary.
     * <p>
     * This method should always be used if the replacement string may contain illegal group
     * references (dollar characters: $).
     *
     * @param text        The text to be replaced
     * @param regex       The regular expressions to be replaced (all occurrences)
     * @param replacement The replacements for the regular expressions, may contain unallowed chars
     * @return The text after replacement
     * @throws NullPointerException                   If <code>text</code>, <code>regex</code>, or <code>replacement</code> is <code>null</code>
     * @throws java.util.regex.PatternSyntaxException If <code>regex</code> is invalid
     */
    public static String safeReplaceAll(String text, String regex, String replacement) {

        replacement = replacement.replaceAll("[\\$]", "%DOLLAR%");  // $db$types.tbl -> %DOLLAR%db%DOLLAR%types.tbl
        text = text.replaceAll(regex, replacement);
        return text.replaceAll("(%DOLLAR%)", "\\$");
    }

    /**
     * Converts the given table data to aligned columns with a certain space.
     *
     * @param array     The array, <code>toString()</code> is called
     * @param spaces    The number of space characters between two columns
     * @param framechar Character to use for the header and footer lines (optional)
     * @return The aligned table
     * @throws NullPointerException If <code>array</code> is <code>null</code>
     */
    public static String assembleTable(Object[][] array, int spaces, Character framechar) {

        if (array.length == 0) return "";     // No rows available
        if (array[0].length == 0) return "";  // No columns available

        StringBuilder builder = new StringBuilder(0);
        String text, linefeed = System.getProperty("line.separator");

        // Determine column widths
        int[] colwidths = new int[array[0].length];

        for (Object[] objects : array) {
            for (int col = 0; col < objects.length; col++) {
                text = String.valueOf(objects[col]);
                colwidths[col] = Math.max(colwidths[col], text.length());
            }
        }
        int totalcols = 0;
        for (int colwidth : colwidths) totalcols += colwidth;
        totalcols += (array[0].length - 1) * spaces;

        // Assemble aligned table now using maximum widths and spaces
        if (framechar != null) {  // Header
            for (int i = 0; i < totalcols; i++) builder.append(framechar);
            builder.append(linefeed);
        }

        for (Object[] objects : array) {
            for (int col = 0; col < objects.length; col++) {
                text = String.valueOf(objects[col]);
                builder.append(text);
                for (int i = text.length(); i < colwidths[col]; i++) builder.append(' ');
                if (col + 1 < objects.length) {
                    for (int i = 0; i < spaces; i++) builder.append(' ');
                }
            }
            builder.append(linefeed);
        }
        if (framechar != null) {  // Footer
            for (int i = 0; i < totalcols; i++) builder.append(framechar);
            builder.append(linefeed);
        }
        return builder.toString();
    }
}
