/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.dropbox;

import com.dropbox.core.v2.DbxClientV2;
import com.dropbox.core.v2.files.ListFolderResult;
import com.lf.vfslib.VFSLib;
import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.net.ClientPool;
import com.lf.vfslib.net.ClientWrapper;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.provider.GenericFileName;

import java.util.logging.Level;


/**
 * Wraps a Dropbox client.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class DbxClientWrapper implements ClientWrapper {


    /**
     * The root file name.
     */
    protected GenericFileName root = null;
    /**
     * The file system options.
     */
    protected FileSystemOptions fileSystemOptions = null;
    /**
     * The parental object.
     */
    protected VFSLib vfsLib = null;
    /**
     * The file provider.
     */
    protected DbxFileProvider provider = null;
    /**
     * The low-level client.
     */
    protected DbxClientV2 dbxClient = null;


    /**
     * Constructor method (do not use).
     *
     * @throws InstantiationException Error indication
     */
    public DbxClientWrapper() throws InstantiationException {

        if (!java.beans.Beans.isDesignTime()) {
            throw new InstantiationException("Do not use this constructor!");
        }
    }

    /**
     * Constructor method.
     *
     * @param root     The file system root
     * @param options  The file system specific configuration
     * @param vfslib   The parental object, shared instance if <code>null</code>
     * @param provider The file provider
     * @throws FileSystemException If something goes wrong
     */
    DbxClientWrapper(GenericFileName root, FileSystemOptions options, VFSLib vfslib, DbxFileProvider provider) throws FileSystemException {

        this.root = root;
        this.fileSystemOptions = options;
        this.vfsLib = vfslib != null ? vfslib : VFSLib.getSharedInstance();
        this.provider = provider;

        createClient();  // Fail fast
    }

    /**
     * Establish connection to Dropbox based on the configured options.
     *
     * @throws FileSystemException If something goes wrong
     */
    protected void createClient() throws FileSystemException {
        try {
            // Create client first
            this.dbxClient = DbxClientFactory.createClient(this.fileSystemOptions, this.provider.getRequestConfig());

            // Fail fast like SFTP: access root URI, provoke exeptions
            ListFolderResult result = this.dbxClient.files().listFolder("");
            result.getEntries().forEach(e -> VFSLibSettings.log(Level.CONFIG, e.getName()));

        } catch (Exception e) {
            String text = VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_CONNECT_ERROR");
            throw new FileSystemException(text.replaceAll("(%server%)", String.valueOf(this.root)), this.root, e);
        }
    }

    /**
     * Provides access to the internal Dropbox client.
     *
     * @return The client reference
     * @since 2.8
     */
    public DbxClientV2 getClient() {
        return this.dbxClient;
    }

    /**
     * Let's the underlying client disconnect from the server.
     *
     * @return Successful?
     */
    // Implements ClientWrapper interface
    public boolean disconnectClient() {
        return true;  // Nothing to do here
    }

    /**
     * Provides the unique URL identifier used to manage this client instance.
     * <p>
     * Something like "dropbox://user@dropbox.com".
     *
     * @return The URL
     */
    // Implements ClientWrapper interface
    public String getURLID() {
        return this.toString();
    }

    /**
     * Gets an idle client from the pool or creates a fresh new instance.
     *
     * @param root     The root path
     * @param options  The file system options
     * @param vfslib   The parental object, shared instance if <code>null</code>
     * @param provider The file provider
     * @return Wrapper instance
     * @throws FileSystemException If an I/O error occurs
     */
    public static DbxClientWrapper getDbxClientWrapper(GenericFileName root, FileSystemOptions options,
                                                       VFSLib vfslib, DbxFileProvider provider) throws FileSystemException {

        if (vfslib == null) vfslib = VFSLib.getSharedInstance();

        // Get connection from pool if possible
        ClientPool pool = ClientPool.getSharedInstance();
        try {
            // dropbox://user@dropbox.com
            ClientWrapper recycled = pool.getFreeClient(getClientURLID(root, options, vfslib));
            if (recycled != null) return (DbxClientWrapper) recycled;
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }

        // Create a fresh new instance
        DbxClientWrapper wrapper = new DbxClientWrapper(root, options, vfslib, provider);
        pool.addClient(wrapper);  // Locked automatically
        return wrapper;
    }

    /**
     * Releases the given client from the pool.
     *
     * @param wrapper The wrapper
     */
    public static void unlockDbxClientWrapper(DbxClientWrapper wrapper) {
        ClientPool.getSharedInstance().unlockClient(wrapper);
    }

    /**
     * Provides a textual representation.
     *
     * @return The text
     */
    @Override
    public String toString() {
        return getClientURLID(this.root, this.fileSystemOptions, this.vfsLib);
    }

    /**
     * Provides a textual representation for this connection.
     *
     * @param root    The root path
     * @param options The file system options
     * @param vfslib  The parental object, shared instance if <code>null</code>
     * @return The text
     */
    public static String getClientURLID(GenericFileName root, FileSystemOptions options, VFSLib vfslib) {

        if (vfslib == null) vfslib = VFSLib.getSharedInstance();

        DbxFileSystemConfigBuilder builder = DbxFileSystemConfigBuilder.getSharedInstance();
        String username = builder.getAccountDisplayName(options);
        if (username == null) username = "anonymous";

        return vfslib.getSchemeDropbox() + "://" + username + '@' + root.getHostName();  // No port
    }
}
