/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib;

// Do not import provider specific classes here, must be usable separately (NoClassDefFoundError)

import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.dropbox.DbxFileProvider;
import com.lf.vfslib.gdrive.GDriveFileProvider;
import com.lf.vfslib.lang.ReflectionUtils;
import com.lf.vfslib.s3.S3FileProvider;
import org.apache.commons.vfs2.impl.DefaultFileSystemManager;

import java.io.File;
import java.io.StringReader;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;


/**
 * Configures and activates the VFSLib providers to access network file systems via Commons VFS.
 * <p>
 * Use the shared instance if you have only one single file system manager instance. Create multiple
 * instances with different configurations with the default constructor methods if there are
 * separate file system managers in use.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class VFSLib {


    /**
     * The shared <code>VFSLib</code> instance.
     */
    protected static VFSLib sharedInstance = null;
    /**
     * The default scheme name for Dropbox "dropbox", override with "dbx" or whatever.
     */
    protected String schemeDropbox = "dropbox";
    /**
     * The default scheme name for Google Drive "gdrive", override with "google" or whatever.
     */
    protected String schemeGoogleDrive = "gdrive";
    /**
     * The default scheme name for Amazon S3 "s3", override with "amazon" or whatever.
     */
    protected String schemeAmazonS3 = "s3";


    /**
     * Constructor method.
     */
    public VFSLib() {
    }

    /**
     * Provides the shared instance of the <code>VFSLib</code>.
     *
     * @return Shared instance
     */

    public static synchronized VFSLib getSharedInstance() {

        if (sharedInstance == null) sharedInstance = new VFSLib();
        return sharedInstance;
    }

    /**
     * Setter method for the VFSLib Dropbox scheme.
     * <p>
     * The default scheme name for Dropbox is "dropbox", override with "dbx" or whatever.
     * This method must be called before the providers are being installed.
     *
     * @param scheme The scheme
     * @throws NullPointerException     If the scheme is <code>null</code> or empty
     * @throws IllegalArgumentException If an error occurs
     */
    public void setSchemeDropbox(String scheme) {

        if (scheme == null) throw new NullPointerException();
        if (scheme.trim().isEmpty()) {
            VFSLibSettings.logMessage("VFSLIB-0013", Level.WARNING);
            throw new IllegalArgumentException(VFSLibSettings.getUserMessage("VFSLIB-0013"));
        }
        this.schemeDropbox = scheme;
    }

    /**
     * Getter method for the VFSLib Dropbox scheme.
     *
     * @return The scheme
     */
    public String getSchemeDropbox() {
        return this.schemeDropbox;
    }

    /**
     * Configures the given file system manager to use the VFSLib provider for Dropbox (convenience method).
     * <p>
     * The required libraries must be in the classpath.
     * Please setup the application info and request configuration for Dropbox first. You may
     * use a different scheme/protocol name by calling <code>setSchemeDropbox()</code> (default is "dropbox").
     *
     * @param manager          The VFSLib manager (required)
     * @param appkey           The Dropbox application info (required for <code>DbxAppInfo</code>)
     * @param appsecret        The Dropbox request configuration (required for <code>DbxAppInfo</code>)
     * @param clientidentifier The Dropbox client identifier (required for <code>DbxRequestConfig</code>)
     * @return The installed provider, <code>null</code> if an error occurred
     * @throws NullPointerException     If a parameter is <code>null</code>
     * @throws IllegalArgumentException If an error occurs
     */
    public DbxFileProvider addProviderDropbox(DefaultFileSystemManager manager, String appkey, String appsecret,
                                              String clientidentifier) {

        if (manager == null) throw new NullPointerException();
        if (appkey == null) throw new NullPointerException();
        if (appsecret == null) throw new NullPointerException();
        if (clientidentifier == null) throw new NullPointerException();

        if (!checkLibsDropbox()) return null;

        // Create Dropbox instances
        //DbxAppInfo appinfo = new DbxAppInfo(appkey, appsecret);
        //DbxRequestConfig reqconfig = new DbxRequestConfig(clientidentifier);

        // Avoid imports, use reflection here
        boolean silent = true; // Set to false only while testing!
        Object appinfo = ReflectionUtils.newInstance(
                "com.dropbox.core.DbxAppInfo",
                new String[]{"java.lang.String", "java.lang.String"}, appkey, appsecret);
        Object reqconfig = ReflectionUtils.newInstance(
                "com.dropbox.core.DbxRequestConfig",
                new String[]{"java.lang.String"}, clientidentifier);

        return this.addProviderDropbox(manager, appinfo, reqconfig);
    }

    /**
     * Configures the given file system manager to use the VFSLib provider for Dropbox.
     * <p>
     * The required libraries must be in the classpath.
     * Please setup the application info and request configuration for Dropbox first. You may
     * use a different scheme/protocol name by calling <code>setSchemeDropbox()</code> (default is "dropbox").
     *
     * @param manager   The VFSLib manager (required)
     * @param appinfo   The Dropbox application info (required, type is <code>com.dropbox.core.DbxAppInfo</code>)
     * @param reqconfig The Dropbox request configuration (required, type is <code>com.dropbox.core.DbxRequestConfig</code>)
     * @return The installed provider, <code>null</code> if an error occurred
     * @throws NullPointerException     If a parameter is <code>null</code>
     * @throws IllegalArgumentException If a parameter is not of the expected type
     */
    public DbxFileProvider addProviderDropbox(DefaultFileSystemManager manager, Object appinfo, Object reqconfig) {

        if (manager == null) throw new NullPointerException();
        if (appinfo == null) throw new NullPointerException();
        if (reqconfig == null) throw new NullPointerException();

        if (!checkLibsDropbox()) return null;

        boolean ok = false;
        DbxFileProvider provider = null;
        try {
            provider = new DbxFileProvider(appinfo, reqconfig, this);
            manager.addProvider(this.schemeDropbox, provider);
            ok = true;
        } catch (Exception e) {
            VFSLibSettings.logMessage("VFSLIB-0017", Level.WARNING);
            System.err.println(VFSLibSettings.getUserMessage("VFSLIB-0017"));
        }
        return ok ? provider : null;
    }

    /**
     * Setter method for the VFSLib Google Drive scheme.
     * <p>
     * The default scheme name for Google Drive is "gdrive", override with "google" or whatever.
     * This method must be called before the providers are being installed.
     *
     * @param scheme The scheme
     * @throws NullPointerException     If the scheme is <code>null</code> or empty
     * @throws IllegalArgumentException If an error occurs
     */
    public void setSchemeGoogleDrive(String scheme) {

        if (scheme == null) throw new NullPointerException();
        if (scheme.trim().isEmpty()) {
            VFSLibSettings.logMessage("VFSLIB-0018", Level.WARNING);
            throw new IllegalArgumentException(VFSLibSettings.getUserMessage("VFSLIB-0018"));
        }
        this.schemeGoogleDrive = scheme;
    }

    /**
     * Getter method for the VFSLib Google Drive scheme.
     *
     * @return The scheme
     */
    public String getSchemeGoogleDrive() {
        return this.schemeGoogleDrive;
    }

    /**
     * Configures the given file system manager to use the VFSLib provider for Google Drive (convenience method).
     * <p>
     * The required libraries must be in the classpath.
     * Please setup a project and application in the Google Cloud Console first. Don't forget to
     * have the Drive API enabled for your project registered from within the Cloud Console.
     * <p/>
     * <p>
     * You may use a different scheme/protocol name by calling <code>getSchemeGoogleDrive()</code>
     * (default is "gdrive").
     *
     * @param manager          The VFSLib manager (required)
     * @param clientsecretjson The OAuth 2.0 client JSON from Google (required)
     * @param appname          The application name (required)
     * @param accesstype       "online" for web applications and "offline" for installed applications (required)
     * @param tokensdir        The directory to store the tokens (optional)
     * @return The installed provider, <code>null</code> if an error occurred
     * @throws NullPointerException     If a parameter is <code>null</code>
     * @throws IllegalArgumentException If an error occurs
     * @since 2.8s
     */
    public GDriveFileProvider addProviderGoogleDrive(DefaultFileSystemManager manager, String clientsecretjson,
                                                     String appname, String accesstype, File tokensdir) {

        if (manager == null) throw new NullPointerException();
        if (clientsecretjson == null) throw new NullPointerException();
        if (appname == null) throw new NullPointerException();

        if (!checkLibsGoogleDrive()) return null;

        // See Java example https://developers.google.com/drive/api/quickstart/java?hl=de
        /*
        GoogleAuthorizationCodeFlow authflow = null;
        try {
            NetHttpTransport httpTransport = GoogleNetHttpTransport.newTrustedTransport();
            JsonFactory jsonFactory = GsonFactory.getDefaultInstance();

            // See, edit, create, and delete all of your Google Drive files
            List<String> scopes = Collections.singletonList(DriveScopes.DRIVE);

            GoogleClientSecrets clientsecrets = GoogleClientSecrets.load(jsonFactory, new StringReader(clientsecretjson));
            DataStoreFactory datastorefactory = (tokensdir != null ? new FileDataStoreFactory(tokensdir) : new MemoryDataStoreFactory());

            // Build flow for user authorization request
            authflow = new GoogleAuthorizationCodeFlow.Builder(httpTransport, jsonFactory, clientsecrets, scopes) //
                    .setDataStoreFactory(datastorefactory) //
                    .setAccessType(accesstype) //
                    .build();
        } catch (Exception ignored) {
        }
         */

        // Avoid imports, use reflection here
        Object httpTransport = ReflectionUtils.invokeStatic(
                "com.google.api.client.googleapis.javanet.GoogleNetHttpTransport", "newTrustedTransport",
                new String[]{});
        Object jsonFactory = ReflectionUtils.invokeStatic(
                "com.google.api.client.json.gson.GsonFactory", "getDefaultInstance",
                new String[]{});

        String scope = (String) ReflectionUtils.getStaticField(
                "com.google.api.services.drive.DriveScopes", "DRIVE");
        List<String> scopes = Collections.singletonList(scope);

        Object clientsecrets = ReflectionUtils.invokeStatic(
                "com.google.api.client.googleapis.auth.oauth2.GoogleClientSecrets", "load",
                new String[]{"com.google.api.client.json.JsonFactory", "java.io.Reader"},
                jsonFactory, new StringReader(clientsecretjson));
        Object datastorefactory = null;
        if (tokensdir != null) {
            datastorefactory = ReflectionUtils.newInstance(
                    "com.google.api.client.util.store.FileDataStoreFactory",
                    new String[]{"java.io.File"}, tokensdir);
        } else {
            datastorefactory = ReflectionUtils.newInstance(
                    "com.google.api.client.util.store.MemoryDataStoreFactory",
                    new String[]{});
        }

        Object authflowbuilder = ReflectionUtils.newInstance(
                "com.google.api.client.googleapis.auth.oauth2.GoogleAuthorizationCodeFlow$Builder",
                new String[]{"com.google.api.client.http.HttpTransport", "com.google.api.client.json.JsonFactory",
                        "com.google.api.client.googleapis.auth.oauth2.GoogleClientSecrets", "java.util.Collection"},
                httpTransport, jsonFactory, clientsecrets, scopes);
        ReflectionUtils.invoke(authflowbuilder, "setDataStoreFactory",
                new String[]{"com.google.api.client.util.store.DataStoreFactory"}, datastorefactory);
        ReflectionUtils.invoke(authflowbuilder, "setAccessType",
                new String[]{"java.lang.String"}, accesstype);
        Object authflow = ReflectionUtils.invoke(authflowbuilder, "build",
                new String[]{});

        return this.addProviderGoogleDrive(manager, authflow, appname);
    }

    /**
     * Configures the given file system manager to use the VFSLib provider for Google Drive.
     * <p/>
     * The required libraries must be in the classpath.
     * Please setup a project and application in the Google Cloud Console first. Don't forget to
     * have the Drive API enabled for your project registered from within the Cloud Console.
     * <p/>
     * You may use a different scheme/protocol name by calling <code>getSchemeGoogleDrive()</code>
     * (default is "gdrive").
     *
     * @param manager  The VFSLib manager (required)
     * @param authflow The Google authorization flow (required, type is <code>com.google.api.client.googleapis.auth.oauth2.GoogleAuthorizationCodeFlow</code>)
     * @param appname  The application name (required)
     * @return The installed provider, <code>null</code> if an error occurred
     * @throws NullPointerException     If a parameter is <code>null</code>
     * @throws IllegalArgumentException If an error occurs
     */
    public GDriveFileProvider addProviderGoogleDrive(DefaultFileSystemManager manager, Object authflow, String appname) {

        if (manager == null) throw new NullPointerException();
        if (authflow == null) throw new NullPointerException();
        if (appname == null) throw new NullPointerException();

        if (!checkLibsGoogleDrive()) return null;

        boolean ok = false;
        GDriveFileProvider provider = null;
        try {
            provider = new GDriveFileProvider(authflow, appname, this);
            manager.addProvider(this.schemeGoogleDrive, provider);
            ok = true;
        } catch (Exception e) {
            VFSLibSettings.logMessage("VFSLIB-0019", Level.WARNING);
            System.err.println(VFSLibSettings.getUserMessage("VFSLIB-0019"));
        }
        return ok ? provider : null;
    }

    /**
     * Setter method for the VFSLib Amazon S3 scheme.
     * <p>
     * The default scheme name for Amazon S3 is "s3", override with "amazon" or whatever.
     * This method must be called before the providers are being installed.
     *
     * @param scheme The scheme
     * @throws NullPointerException     If the scheme is <code>null</code> or empty
     * @throws IllegalArgumentException If an error occurs
     */
    public void setSchemeAmazonS3(String scheme) {

        if (scheme == null) throw new NullPointerException();
        if (scheme.trim().isEmpty()) {
            VFSLibSettings.logMessage("VFSLIB-0022", Level.WARNING);
            throw new IllegalArgumentException(VFSLibSettings.getUserMessage("VFSLIB-0022"));
        }
        this.schemeAmazonS3 = scheme;
    }

    /**
     * Getter method for the VFSLib Amazon S3 scheme.
     *
     * @return The scheme
     */
    public String getSchemeAmazonS3() {
        return this.schemeAmazonS3;
    }

    /**
     * Configures the given file system manager to use the VFSLib provider for Amazon S3 (convenience method).
     * <p>
     * The required libraries must be in the classpath.
     * Please setup a valid Amazon Web Services (AWS) account first. You may
     * use a different scheme/protocol name by calling <code>setSchemeAmazonS3()</code> (default is "s3").
     *
     * @param manager The VFSLib manager (required)
     * @return The installed provider, <code>null</code> if an error occurred
     * @throws NullPointerException     If a parameter is <code>null</code>
     * @throws IllegalArgumentException If an error occurs
     */
    public S3FileProvider addProviderAmazonS3(DefaultFileSystemManager manager) {

        if (manager == null) throw new NullPointerException();

        if (!checkLibsAmazonS3()) return null;

        boolean ok = false;
        S3FileProvider provider = null;
        try {
            provider = new S3FileProvider(this);
            manager.addProvider(this.schemeAmazonS3, provider);
            ok = true;
        } catch (Exception e) {
            VFSLibSettings.logMessage("VFSLIB-0023", Level.WARNING);
            System.err.println(VFSLibSettings.getUserMessage("VFSLIB-0023"));
        }
        return ok ? provider : null;
    }

    /**
     * Checks if the required libraries are accessible.
     *
     * @return Available?
     */
    protected static boolean checkLibsAmazonS3() {

        try {  // Throws
            Class.forName("software.amazon.awssdk.services.s3.model.S3Object");
            return true;
        } catch (Exception e) {
            VFSLibSettings.logMessage("VFSLIB-0024", Level.WARNING);
        }
        return false;
    }

    /**
     * Checks if the required libraries are accessible.
     *
     * @return Available?
     */
    protected static boolean checkLibsDropbox() {

        try {  // Throws
            Class.forName("com.dropbox.core.v2.files.FileMetadata");
            return true;
        } catch (Exception e) {
            VFSLibSettings.logMessage("VFSLIB-0016", Level.WARNING);
        }
        return false;
    }

    /**
     * Checks if the required libraries are accessible.
     *
     * @return Available?
     */
    protected static boolean checkLibsGoogleDrive() {

        try {  // Throws
            Class.forName("com.google.api.services.drive.model.File");
            return true;
        } catch (Exception e) {
            VFSLibSettings.logMessage("VFSLIB-0020", Level.WARNING);
        }
        return false;
    }
}
