/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.s3;

import org.apache.commons.vfs2.FileSystem;
import org.apache.commons.vfs2.FileSystemConfigBuilder;
import org.apache.commons.vfs2.FileSystemOptions;


/**
 * The config builder for the various Amazon S3 configuration options.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class S3FileSystemConfigBuilder extends FileSystemConfigBuilder {


    /**
     * Shared <code>DbxFileSystemConfigBuilder</code> instance.
     */
    protected static S3FileSystemConfigBuilder sharedInstance;

    /**
     * Display name.
     */
    public final static String DISPLAY_NAME = S3FileSystemConfigBuilder.class.getName() + ".DISPLAY_NAME";
    /**
     * Access key ID.
     */
    public final static String ACCESS_KEY_ID = S3FileSystemConfigBuilder.class.getName() + ".ACCESS_KEY_ID";
    /**
     * Secret key.
     */
    public final static String SECRET_KEY = S3FileSystemConfigBuilder.class.getName() + ".SECRET_KEY";
    /**
     * Bucket URI.
     */
    public final static String AWS_REGION = S3FileSystemConfigBuilder.class.getName() + ".AWS_REGION";
    /**
     * Bucket name.
     */
    public final static String BUCKET_NAME = S3FileSystemConfigBuilder.class.getName() + ".BUCKET_NAME";


    /**
     * Constructor method.
     *
     * @since 1.6
     */
    public S3FileSystemConfigBuilder() {
        super("s3.");  // Static to find resources, do not use dynamic scheme here
    }

    /**
     * Provides the shared instance of the factory implementation.
     *
     * @return Shared instance
     * @since 1.6
     */
    public static synchronized S3FileSystemConfigBuilder getSharedInstance() {

        if (sharedInstance == null) sharedInstance = new S3FileSystemConfigBuilder();
        return sharedInstance;
    }

    /**
     * Provides the file system class to be configured by this builder.
     *
     * @return File system class
     * @since 1.6
     */
    @Override
    protected Class<? extends FileSystem> getConfigClass() {
        return S3FileSystem.class;
    }

    /**
     * Setter for Amazon S3 account display name.
     *
     * @param options The options
     * @param name    The display name
     * @since 1.6
     */
    public void setAccountDisplayName(FileSystemOptions options, String name) {
        super.setParam(options, DISPLAY_NAME, name);
    }

    /**
     * Getter for Amazon S3 account display name.
     *
     * @param options The options
     * @return The display name, may be <code>null</code>
     * @since 1.6
     */
    public String getAccountDisplayName(FileSystemOptions options) {
        return super.getParam(options, DISPLAY_NAME);
    }

    /**
     * Setter for Amazon S3 access key ID.
     *
     * @param options The options
     * @param keyid   The access key ID
     * @since 1.6
     */
    public void setAccessKeyID(FileSystemOptions options, String keyid) {
        super.setParam(options, ACCESS_KEY_ID, keyid);
    }

    /**
     * Getter for Amazon S3 access key ID.
     *
     * @param options The options
     * @return The access key ID, may be <code>null</code>
     * @since 1.6
     */
    public String getAccessKeyID(FileSystemOptions options) {
        return super.getParam(options, ACCESS_KEY_ID);
    }

    /**
     * Setter for Amazon S3 secret key.
     *
     * @param options The options
     * @param key     The secret key
     * @since 1.6
     */
    public void setSecretKey(FileSystemOptions options, String key) {
        super.setParam(options, SECRET_KEY, key);
    }

    /**
     * Getter for Amazon S3 secret key.
     *
     * @param options The options
     * @return The secret key, may be <code>null</code>
     * @since 1.6
     */
    public String getSecretKey(FileSystemOptions options) {
        return super.getParam(options, SECRET_KEY);
    }

    /**
     * Setter for the AWS region for the bucket.
     *
     * @param options The options
     * @param region  The region
     * @since 2.8
     */
    public void setAWSRegion(FileSystemOptions options, String region) {
        super.setParam(options, AWS_REGION, region);
    }

    /**
     * Getter for the AWS region for the bucket.
     *
     * @param options The options
     * @return The region. May be <code>null</code>.
     * @since 2.8
     */
    public String getAWSRegion(FileSystemOptions options) {
        return super.getParam(options, AWS_REGION);
    }

    /**
     * Setter for the name of the Amazon S3 bucket.
     *
     * @param options The options
     * @param name    The display name
     * @since 1.6
     */
    public void setBucketName(FileSystemOptions options, String name) {
        super.setParam(options, BUCKET_NAME, name);
    }

    /**
     * Getter for the name of the Amazon S3 bucket.
     *
     * @param options The options
     * @return The display name, may be <code>null</code>
     * @since 1.6
     */
    public String getBucketName(FileSystemOptions options) {
        return super.getParam(options, BUCKET_NAME);
    }
}
