/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.s3;

import com.lf.vfslib.VFSLib;
import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.net.ClientPool;
import com.lf.vfslib.net.ClientWrapper;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.provider.GenericFileName;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.Bucket;
import software.amazon.awssdk.services.s3.model.ListBucketsResponse;

import java.util.List;
import java.util.logging.Level;


/**
 * Wraps a Amazon S3 client.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class S3ClientWrapper implements ClientWrapper {


    /**
     * The root file name.
     */
    protected GenericFileName root = null;
    /**
     * The file system options.
     */
    protected FileSystemOptions fileSystemOptions = null;
    /**
     * The parental object.
     */
    protected VFSLib vfsLib = null;
    /**
     * The file provider.
     */
    protected S3FileProvider provider = null;
    /**
     * The low-level client.
     */
    protected S3Client s3Client = null;


    /**
     * Constructor method (do not use).
     *
     * @throws InstantiationException Error indication
     * @since 1.6
     */
    public S3ClientWrapper() throws InstantiationException {

        if (!java.beans.Beans.isDesignTime()) {
            throw new InstantiationException("Do not use this constructor!");
        }
    }

    /**
     * Constructor method.
     *
     * @param root     The file system root
     * @param options  The file system specific configuration
     * @param vfslib   The parental object, shared instance if <code>null</code>
     * @param provider The file provider
     * @throws org.apache.commons.vfs2.FileSystemException If something goes wrong
     * @since 1.6
     */
    S3ClientWrapper(GenericFileName root, FileSystemOptions options, VFSLib vfslib, S3FileProvider provider)
            throws FileSystemException {

        this.root = root;
        this.fileSystemOptions = options;
        this.vfsLib = vfslib != null ? vfslib : VFSLib.getSharedInstance();
        this.provider = provider;

        createClient();  // Fail fast
    }

    /**
     * Establish connection to Amazon S3 based on the configured options.
     *
     * @throws org.apache.commons.vfs2.FileSystemException If something goes wrong
     * @since 1.6
     */
    protected void createClient() throws FileSystemException {
        try {
            // Create client first

            // The credentials may differ for each connection, use access key ID internally, not the user name
            S3FileSystemConfigBuilder builder = S3FileSystemConfigBuilder.getSharedInstance();
            String accesskeyid = builder.getAccessKeyID(this.fileSystemOptions);  // Provided by Amazon
            String secretkey = builder.getSecretKey(this.fileSystemOptions);  // Provided by Amazon

            AwsCredentials credentials = AwsBasicCredentials.create(accesskeyid, secretkey);
            StaticCredentialsProvider provider = StaticCredentialsProvider.create(credentials);

            this.s3Client = S3ClientFactory.createClient(this.root, this.fileSystemOptions, provider, this.vfsLib);

            // Fail fast like SFTP: list buckets, provoke execptions
            ListBucketsResponse response = this.s3Client.listBuckets();
            List<Bucket> buckets = response.buckets();
            buckets.forEach(b -> VFSLibSettings.log(Level.CONFIG, b.name()));

        } catch (Exception e) {
            String text = VFSLibSettings.getUserText(S3ClientWrapper.class.getName() + "_CONNECT_ERROR");
            throw new FileSystemException(text.replaceAll("(%server%)", String.valueOf(this.root)), this.root, e);
        }
    }

    /**
     * Provides access to the internal Amazon S3 client.
     *
     * @return The client reference
     * @since 1.6
     */
    public S3Client getClient() {
        return this.s3Client;
    }

    /**
     * Let's the underlying client disconnect from the server.
     *
     * @return Successful?
     * @since 1.6
     */
    // Implements ClientWrapper interface
    public boolean disconnectClient() {
        return true;  // Nothing to do here
    }

    /**
     * Provides the unique URL identifier used to manage this client instance.
     * <p>
     * Something like "s3://user@aws.amazon.com".
     *
     * @return The URL
     * @since 1.6
     */
    // Implements ClientWrapper interface
    public String getURLID() {
        return this.toString();
    }

    /**
     * Gets an idle client from the pool or creates a fresh new instance.
     *
     * @param root     The root path
     * @param options  The file system options
     * @param vfslib   The parental object, shared instance if <code>null</code>
     * @param provider The file provider
     * @return Wrapper instance
     * @throws org.apache.commons.vfs2.FileSystemException If an I/O error occurs
     * @since 1.6
     */
    public static S3ClientWrapper getS3ClientWrapper(GenericFileName root, FileSystemOptions options,
                                                     VFSLib vfslib, S3FileProvider provider) throws FileSystemException {

        if (vfslib == null) vfslib = VFSLib.getSharedInstance();

        // Get connection from pool if possible
        ClientPool pool = ClientPool.getSharedInstance();
        try {
            // s3://user@aws.amazon.com
            ClientWrapper recycled = pool.getFreeClient(getClientURLID(root, options, vfslib));
            if (recycled != null) return (S3ClientWrapper) recycled;
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }

        // Create a fresh new instance
        S3ClientWrapper wrapper = new S3ClientWrapper(root, options, vfslib, provider);
        pool.addClient(wrapper);  // Locked automatically
        return wrapper;
    }

    /**
     * Releases the given client from the pool.
     *
     * @param wrapper The wrapper
     * @since 1.6
     */
    public static void unlockS3ClientWrapper(S3ClientWrapper wrapper) {
        ClientPool.getSharedInstance().unlockClient(wrapper);
    }

    /**
     * Provides a textual representation.
     *
     * @return The text
     * @since 1.6
     */
    @Override
    public String toString() {
        return getClientURLID(this.root, this.fileSystemOptions, this.vfsLib);
    }

    /**
     * Provides a textual representation for this connection.
     *
     * @param root    The root path
     * @param options The file system options
     * @param vfslib  The parental object, shared instance if <code>null</code>
     * @return The text
     * @since 1.6
     */
    public static String getClientURLID(GenericFileName root, FileSystemOptions options, VFSLib vfslib) {

        if (vfslib == null) vfslib = VFSLib.getSharedInstance();

        S3FileSystemConfigBuilder builder = S3FileSystemConfigBuilder.getSharedInstance();
        String username = builder.getAccountDisplayName(options);
        if (username == null) username = "anonymous";

        return vfslib.getSchemeAmazonS3() + "://" + username + '@' + root.getHostName();  // No port
    }
}
