/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.net;

import com.lf.vfslib.VFSLib;
import com.lf.vfslib.core.VFSLibConstants;
import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.io.IOUtils;
import com.lf.vfslib.lang.JavaUtils;
import com.lf.vfslib.lang.ReflectionUtils;
import org.apache.commons.vfs2.*;

import javax.swing.*;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.util.*;
import java.util.logging.Level;


/**
 * This class holds various methods for common use with Apache Commons VFS networking.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class VFSUtils {


    /**
     * Constructor method for i18n purposes only.
     *
     * @throws InstantiationException Error indication
     * @since 1.6
     */
    public VFSUtils() throws InstantiationException {

        if (!java.beans.Beans.isDesignTime()) {
            throw new InstantiationException("Do not use this constructor!");
        }
    }

    /**
     * Safely gets the URL content length.
     *
     * @param url The network resource to check
     * @return The length or <code>null</code> if not available
     * @since 1.6
     */
    public static Long getLength(FileObject url) {

        try {
            return url.getContent().getSize();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the URL content length.
     *
     * @param url      The network resource to check
     * @param fallback The default value
     * @return The length or <code>null</code> if not available
     * @since 1.6
     */
    public static long getLength(FileObject url, long fallback) {

        try {
            return url.getContent().getSize();
        } catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Safely gets the URL's last modifcation time.
     *
     * @param url The network resource to check
     * @return The last modification time or <code>null</code> if not available
     * @since 1.6
     */
    public static Long lastModified(FileObject url) {

        try {
            return url.getContent().getLastModifiedTime();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the URL's last modifcation time.
     *
     * @param url      The network resource to check
     * @param fallback The default value
     * @return The last modification time or <code>null</code> if not available
     * @since 1.6
     */
    public static long lastModified(FileObject url, long fallback) {

        try {
            return url.getContent().getLastModifiedTime();
        } catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Safely sets the URL's last modifcation time.
     *
     * @param url       The network resource to check
     * @param timestamp The timestamp
     * @since 1.6
     */
    public static void setLastModified(FileObject url, long timestamp) {

        try {
            url.getContent().setLastModifiedTime(timestamp);
        } catch (Exception ignored) {
        }
    }

    /**
     * Safely gets the info if the URL is a folder/directory.
     *
     * @param url The network resource to check
     * @return Directory? <code>null</code> if not available.
     * @since 1.6
     */
    public static Boolean isDirectory(FileObject url) {

        try {
            // Caution: Also IMAGINARY may be a directory (see "ftp://mycompany.com/usr/local/ftp/pub")
            return !url.getType().equals(FileType.FILE);
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the info if the URL is a folder/directory.
     *
     * @param url      The network resource to check
     * @param fallback The default value
     * @return Directory? <code>null</code> if not available.
     * @since 1.6
     */
    public static boolean isDirectory(FileObject url, boolean fallback) {

        try {
            // Caution: Also IMAGINARY may be a directory (see "ftp://mycompany.com/usr/local/ftp/pub")
            return !url.getType().equals(FileType.FILE);
        } catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Safely gets the info if the URL is a folder/directory.
     *
     * @param vfsuri   The network resource to check
     * @param fallback The default value
     * @return Directory? <code>null</code> if not available.
     * @since 1.6
     */
    public static boolean isDirectory(VFSURI vfsuri, boolean fallback) {

        try {
            // Caution: Also IMAGINARY may be a directory (see "ftp://mycompany.com/usr/local/ftp/pub")
            return !vfsuri.getFileType().equals(FileType.FILE);  // Optional payload
        } catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Safely gets the info if the URL is a file.
     *
     * @param url The network resource to check
     * @return File? <code>null</code> if not available.
     * @since 1.6
     */
    public static Boolean isFile(FileObject url) {

        try {
            return url.getType().equals(FileType.FILE);
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the info if the URL is a file.
     *
     * @param url      The network resource to check
     * @param fallback The default value
     * @return File? <code>null</code> if not available.
     * @since 1.6
     */
    public static boolean isFile(FileObject url, boolean fallback) {

        try {
            return url.getType().equals(FileType.FILE);
        } catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Safely gets the info if the URL is a file.
     *
     * @param vfsuri   The network resource to check
     * @param fallback The default value
     * @return File? <code>null</code> if not available.
     * @since 1.6
     */
    public static boolean isFile(VFSURI vfsuri, boolean fallback) {

        try {
            return vfsuri.getFileType().equals(FileType.FILE);
        }  // Optional payload
        catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Safely gets the entry type.
     *
     * @param url The network resource to check
     * @return The type, <code>null</code> if not available.
     * @since 2.8
     */
    public static FileType getType(FileObject url) {

        try {
            return url.getType();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the info if the URL exists.
     *
     * @param url The network resource to check
     * @return Exists? <code>null</code> if not available.
     * @since 1.6
     */
    public static Boolean exists(FileObject url) {

        try {
            return url.exists();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the info if the URL exists.
     *
     * @param url      The network resource to check
     * @param fallback The default value
     * @return Exists? <code>null</code> if not available.
     * @since 1.6
     */
    public static boolean exists(FileObject url, boolean fallback) {

        try {
            return url.exists();
        } catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Safely gets the info if the URL can be read.
     *
     * @param url The network resource to check
     * @return Readable? <code>null</code> if not available.
     * @since 1.6
     */
    public static Boolean canRead(FileObject url) {

        try {
            return url.isReadable();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the info if the URL can be read.
     *
     * @param url      The network resource to check
     * @param fallback The default value
     * @return Readable? <code>null</code> if not available.
     * @since 1.6
     */
    public static boolean canRead(FileObject url, boolean fallback) {

        try {
            return url.isReadable();
        } catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Safely gets the info if the URL can be written.
     *
     * @param url The network resource to check
     * @return Writable? <code>null</code> if not available.
     * @since 1.6
     */
    public static Boolean canWrite(FileObject url) {

        try {
            return url.isWriteable();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the info if the URL can be written.
     *
     * @param url      The network resource to check
     * @param fallback The default value
     * @return Writable? <code>null</code> if not available.
     * @since 1.6
     */
    public static boolean canWrite(FileObject url, boolean fallback) {

        try {
            return url.isWriteable();
        } catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Safely gets the info if the URL file system has a certain capability.
     *
     * @param url        The network resource to check
     * @param capability The capability
     * @return Available? <code>null</code> if not available.
     * @since 1.6
     */
    public static Boolean hasCapability(FileObject url, Capability capability) {

        try {
            return url.getFileSystem().hasCapability(capability);
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the info if the URL file system has a certain capability.
     *
     * @param url        The network resource to check
     * @param capability The capability
     * @param fallback   The default value
     * @return Available? <code>null</code> if not available.
     * @since 1.6
     */
    public static boolean hasCapability(FileObject url, Capability capability, boolean fallback) {

        try {
            return url.getFileSystem().hasCapability(capability);
        } catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Safely gets the URL parent folder.
     *
     * @param url The network resource to check
     * @return The parent folder or <code>null</code> if not available
     * @since 1.6
     */
    public static FileObject getParent(FileObject url) {

        try {
            return url.getParent();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the URL assembled of parent and relative path.
     *
     * @param vfsmanager The file system manager, use default if <code>null</code>
     * @param parenturl  The parental network resource (folder)
     * @param relpath    The relative path
     * @return The assembled file or <code>null</code> if not available
     * @since 1.6
     */
    public static FileObject getRelativeFileURL(FileSystemManager vfsmanager, FileObject parenturl, String relpath) {

        try {
            // Parent:          http://mycompany.com/
            // Relative path:   installation.pdf
            // Candidate:       http://mycompany.com/installation.pdf
            String path = parenturl.toString() + (parenturl.toString().endsWith("/") ? "" : '/') + relpath;
            if (vfsmanager == null) vfsmanager = VFSLibSettings.getManagerNetwork();

            FileObject reltarget = vfsmanager.resolveFile(path);
            if (reltarget.exists() && reltarget.getType().equals(FileType.FILE)) return reltarget;
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Assembles the preferred target file name for downloads.
     *
     * @param directory The target directory
     * @param url       The network resource to download
     * @return The preferred file or <code>null</code> if not available
     * @since 1.6
     */
    public static File getDownloadFile(File directory, FileObject url) {

        try {
            // Directory:    C:\Temp
            // URL:          http://mycompany.com/installation.pdf
            // Candidate:    C:\Temp\installation.pdf
            return new File(directory.getAbsoluteFile() + File.separator + url.getName().getBaseName());
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Opens an input stream to read URL data either from cache or directly.
     * <p>
     * Caution: Do not use Hashtable<FileObject, ... since FileObject's for the same folder
     * use different hash codes. Such folders are then added multiple times and not once as expected!
     *
     * @param url      The network resource to check
     * @param urlcache The download cache
     * @param maxsize  The maximum file size to be cached
     * @return The input stream or <code>null</code> if not available
     * @since 1.6
     */
    public static InputStream openCachedURL(FileObject url, TreeMap<FileObject, byte[]> urlcache, long maxsize) {

        InputStream istream;

        try {
            if (urlcache == null) return url.getContent().getInputStream();

            // If possible use data from cache
            byte[] data = urlcache.get(url);
            if (data != null) {
                istream = new ByteArrayInputStream(data);
                VFSLibSettings.log(Level.CONFIG, "Recycled URL data for " + url);
                return istream;
            }

            // Download data and put into cache if size fits
            if (url.getContent().getSize() <= maxsize) {
                data = IOUtils.readData(url);
                urlcache.put(url, data);
                return new ByteArrayInputStream(data);
            } else return url.getContent().getInputStream();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Creates a VFS network resource object for the given path which should be a URL.
     *
     * @param vfsmanager The file system manager, use default if <code>null</code>
     * @param url        The network resource path
     * @return The URL object or <code>null</code> if not available
     * @since 1.6
     */
    public static FileObject getURL(FileSystemManager vfsmanager, String url) {
        try {
            // Try to resolve the given URL, protocol may be unsupported etc.
            if (vfsmanager == null) vfsmanager = VFSLibSettings.getManagerNetwork();
            return vfsmanager.resolveFile(url);
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Creates a VFS network resource object for the given path which should be a URL.
     *
     * @param vfsmanager The file system manager, use default if <code>null</code>
     * @param url        The network resource path
     * @param vfsconn    The connection (optional, best guess otherwise)
     * @return The URL object or <code>null</code> if not available
     * @since 1.6
     */
    public static FileObject getURL(VFSFileSystemManager vfsmanager, String url, VFSConnection vfsconn) {

        try {
            // Try to resolve the given URL, protocol may be unsupported etc.
            if (vfsmanager == null) vfsmanager = VFSLibSettings.getManagerNetwork();
            return vfsmanager.resolveFile(url, vfsconn);
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Safely gets the info if the URL is currently available (like a PING check).
     * <p>
     * This method works well for SFTP but not for FTP.
     *
     * @param url The network resource to check
     * @return Available?
     * @since 1.6
     */
    public static boolean available(FileObject url) {

        try {
            return url.exists();  // Do not list children here!
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Determines whether the given URL is a child of the given parent.
     *
     * @param url    The URL to be checked
     * @param parent The parent URL to be checked
     * @return Is it child?
     * @since 1.6
     */
    public static boolean isChildOf(FileObject url, FileObject parent) {

        try {
            FileObject nextparent = url.getParent();
            while (nextparent != null) {
                if (nextparent.equals(parent)) return true;
                nextparent = nextparent.getParent();
            }
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Determines whether the given URL is a child of the given parent.
     *
     * @param url    The URL to be checked
     * @param parent The parent URL to be checked
     * @return Is it child?
     * @since 1.6
     */
    public static boolean isChildOf(VFSURI url, VFSURI parent) {

        try {
            VFSURI nextparent = getParent(url);
            while (nextparent != null) {
                if (nextparent.equals(parent)) return true;
                nextparent = getParent(nextparent);
            }
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Provides the parent directory of the given URL.
     *
     * @param child The child URL
     * @return The parent URL object, <code>null</code> for root
     * @since 1.6
     */
    public static VFSURI getParent(VFSURI child) {

        // sftp://mycompany.com:666/home/aschwolow
        // sftp://mycompany.com:666/home
        // sftp://mycompany.com:666/

        try {
            String[] parts = child.getPath().split("[/\\\\]");
            if (parts.length >= 2) {
                StringBuilder builder = new StringBuilder(0);
                for (int i = 0; i < parts.length - 1; i++) {
                    builder.append(parts[i]);
                    builder.append('/');
                }
                return new VFSURI(child.getScheme(),
                        child.getUserInfo(),
                        child.getHost(),
                        child.getPort(),
                        builder.toString(),
                        child.getQuery(),
                        child.getFragment());
            }
        } catch (Exception ignored) {
        }
        return null;  // Must be root
    }

    /**
     * Generates the accessibility string ("rwx") for network files and directories.
     *
     * @param url The URL to be checked
     * @return Accessibility string e. g. "rwx" or "-r-"
     * @since 1.6
     */
    public static String getRWX(FileObject url) {

        // There is no canExecute() possible...
        return String.valueOf((canRead(url, false) ? 'r' : '-')) + (canWrite(url, false) ? 'w' : '-') + 'x';
    }

    /**
     * Generates the accessibility string ("drwx") for files and directories.
     *
     * @param url The URL to be checked
     * @return Accessibility string e. g. "drwx" (directory) or "--r-" (file)
     * @since 1.6
     */
    public static String getDRWX(FileObject url) {
        return (isDirectory(url, false) ? 'd' : '-') + getRWX(url);
    }

    /**
     * Provides the base name of the given URL (without path, credentials etc.).
     * <p>
     * Special server encodings are converted here if the connection is specified.
     *
     * @param url     The URL
     * @param vfsconn The connection for encoding (optional)
     * @return The display name
     * @since 1.6
     */
    public static String getDisplayBaseName(FileObject url, VFSConnection vfsconn) {

        try {
            return url.getName().getBaseName();
        } catch (Exception ignored) {
        }
        return "null";
    }

    /**
     * Provides the path of the given URL.
     * <p>
     * Special server encodings are converted here if the connection is specified.
     *
     * @param url     The URL
     * @param vfsconn The connection for encoding (optional)
     * @return The display name
     * @since 1.6
     */
    public static String getDisplayPath(FileObject url, VFSConnection vfsconn) {

        try {
            return url.getName().getPath();
        } catch (Exception ignored) {
        }
        return "null";
    }

    /**
     * Provides a name without passwords to be displayed, logged etc.
     * <p>
     * Remember that exception stack traces also may contain passwords as clear text!
     * Special server encodings are converted here if the connection is specified.
     *
     * @param url     The URL
     * @param vfsconn The connection for encoding (optional)
     * @return The display name
     * @since 1.6
     */
    public static String getDisplayURIObfuscated(FileObject url, VFSConnection vfsconn) {

        try {
            return url.getName().getFriendlyURI();
        } catch (Exception ignored) {
        }
        return "null";
    }

    /**
     * Provides a name without passwords to be displayed, logged etc.
     * <p>
     * This variant shows the username if it is available, but never the password.
     * Special server encodings are converted here if the connection is specified.
     *
     * @param entry   The URL
     * @param vfsconn The connection for encoding (optional)
     * @return The display name
     * @since 1.6
     */
    public static String getDisplayURIUsernameOnly(FileObject entry, VFSConnection vfsconn) {
        return getDisplayURIUsernameOnly(String.valueOf(entry), vfsconn);
    }

    /**
     * Provides a name without passwords to be displayed, logged etc.
     * <p>
     * This variant shows the username if it is available, but never the password.
     * Special server encodings are converted here if the connection is specified.
     *
     * @param entry   The URL
     * @param vfsconn The connection for encoding (optional)
     * @return The display name
     * @since 1.6
     */
    public static String getDisplayURIUsernameOnly(VFSURI entry, VFSConnection vfsconn) {
        return getDisplayURIUsernameOnly(String.valueOf(entry), vfsconn);
    }

    /**
     * Provides a name without passwords to be displayed, logged etc.
     * <p>
     * This variant shows the username if it is available, but never the password.
     * Special server encodings are converted here if the connection is specified.
     *
     * @param url     The URL
     * @param vfsconn The connection for encoding (optional)
     * @return The display name
     * @since 1.6
     */
    public static String getDisplayURIUsernameOnly(String url, VFSConnection vfsconn) {

        try {
            // Example:
            // ftp://[ username [: password ]@] hostname [: port ][ absolute-path ]
            String name = url;

            // Caution: Do not use URI here, since included SPACEs etc. lead to exception (OK for VFS)!
            // Caution: Do not use URL here, since "sftp" is not a supported protocol (exception)!
            String scheme = new VFSURI(url).getScheme();
            name = name.replaceFirst("(^" + scheme + "://)([^:]+)(:[^@]+)(@.*$)", "$1$2$4");
            return name;
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, "Malformed URL " + url);
        }
        return "null";
    }

    /**
     * Provides a name without passwords to be displayed, logged etc.
     * <p>
     * Remember that exception stack traces also may contain passwords as clear text!
     * Special server encodings are converted here if the connection is specified.
     *
     * @param url     The URL
     * @param vfsconn The connection for encoding (optional)
     * @return The display name
     * @since 1.6
     */
    public static String getDisplayURIWithoutCredentials(FileObject url, VFSConnection vfsconn) {

        try {
            return getDisplayURIWithoutCredentials(String.valueOf(url), url.getName().getScheme(), vfsconn);
        } catch (Exception ignored) {
        }
        return "null";
    }

    /**
     * Provides a name without passwords to be displayed, logged etc.
     * <p>
     * Remember that exception stack traces also may contain passwords as clear text!
     * Special server encodings are converted here if the connection is specified.
     *
     * @param vfsuri  The URL
     * @param vfsconn The connection for encoding (optional)
     * @return The display name
     * @since 1.6
     */
    public static String getDisplayURIWithoutCredentials(VFSURI vfsuri, VFSConnection vfsconn) {

        try {
            return getDisplayURIWithoutCredentials(String.valueOf(vfsuri), vfsuri.getScheme(), vfsconn);
        } catch (Exception ignored) {
        }
        return "null";
    }

    /**
     * Provides a name without passwords to be displayed, logged etc.
     * <p>
     * Remember that exception stack traces also may contain passwords as clear text!
     * Special server encodings are converted here if the connection is specified.
     *
     * @param path    The URL
     * @param scheme  The protocol
     * @param vfsconn The connection for encoding (optional)
     * @return The display name
     * @since 1.6
     */
    protected static String getDisplayURIWithoutCredentials(String path, String scheme, VFSConnection vfsconn) {

        try {
            // Example:
            // ftp://[ username [: password ]@] hostname [: port ][ absolute-path ]
            return path.replaceFirst("(^" + scheme + "://)([^:]+)(:[^@]+@)(.*$)", "$1$4");
        } catch (Exception ignored) {
        }
        return "null";
    }

    /**
     * Provides a name without passwords to be displayed, logged etc.
     *
     * @param url The URL
     * @return The text
     * @since 1.6
     */
    public static String debugPrintFileName(FileObject url) {

        StringBuilder builder = new StringBuilder(0);
        FileName name = url.getName();

        try {
            builder.append("Base name:    ").append(name.getBaseName()).append('\n');
            builder.append("Depth:        ").append(name.getDepth()).append('\n');
            builder.append("Extension:    ").append(name.getExtension()).append('\n');
            builder.append("Friendly URL: ").append(name.getFriendlyURI()).append('\n');
            builder.append("Path:         ").append(name.getPath()).append('\n');
            builder.append("Path decoded: ").append(name.getPathDecoded()).append('\n');
            builder.append("Parent:       ").append(name.getParent()).append('\n');
            builder.append("Root:         ").append(name.getRoot()).append('\n');
            builder.append("Root URI:     ").append(name.getRootURI()).append('\n');
            builder.append("Scheme:       ").append(name.getScheme()).append('\n');
            builder.append("Type:         ").append(name.getType()).append('\n');
            builder.append("URI :         ").append(name.getURI()).append('\n');
        } catch (Exception ignored) {
        }
        return builder.toString();
    }

    /**
     * Determines whether the given URL is a the parent of the given child.
     *
     * @param url   The URL to be checked
     * @param child The child URL to be checked
     * @return It it parent?
     * @throws NullPointerException If a parameter is <code>null</code>
     * @since 1.6
     */
    public static boolean isParentOf(FileObject url, FileObject child) {

        try {
            FileObject nextparent = child.getParent();
            while (nextparent != null) {
                if (nextparent.equals(url)) return true;
                nextparent = nextparent.getParent();
            }
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Extracts the parent paths of the given URL.
     *
     * @param url The URL to be checked
     * @return Container with parents, direct parent at index 0
     * @throws NullPointerException If the file is <code>null</code>
     * @since 1.6
     */
    public static Vector<FileObject> getParents(FileObject url) {

        Vector<FileObject> parents = new Vector<FileObject>(0);

        try {
            FileObject nextparent = url.getParent();
            while (nextparent != null) {
                parents.addElement(nextparent);
                nextparent = nextparent.getParent();
            }
        } catch (Exception ignored) {
        }
        return parents;
    }

    /**
     * Provides the user language description of the given file system capability.
     *
     * @param capa The capability value
     * @return The capability description, <code>null</code> otherwise
     * @since 1.6
     */
    public static String getCapabilityDescription(Capability capa) {
        try {
            return VFSLibSettings.getUserText(VFSUtils.class.getName() + "_CAPA_" + capa);
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Provides the children found in a directory.
     * <p>
     * Since <code>FileObject.getChildren()</code> may take very long, this method warns with a stack
     * trace if this code is called from the event displatch thread.  Application developers
     * should call this method instead of <code>FileObject.getChildren()</code> to be warned.
     *
     * @param directory The directory
     * @return The childs
     * @throws FileSystemException If an error occurs
     * @since 1.6
     */
    public static FileObject[] getChildren(FileObject directory) throws FileSystemException {

        if (SwingUtilities.isEventDispatchThread()) {
            VFSLibSettings.log(Level.WARNING, "Unsafe: Do not call getChildren() from the EDT (may block the GUI)!");
            Thread.dumpStack();
        }
        return directory.getChildren();
    }

    /**
     * Provides the children found in a directory.
     * <p>
     * Since <code>FileObject.getChildren()</code> may take very long, this method warns with a stack
     * trace if this code is called from the event displatch thread.  Application developers
     * should call this method instead of <code>FileObject.getChildren()</code> to be warned.
     *
     * @param directory The directory
     * @param fallback  The default value
     * @return The childs, fallback otherwise
     * @since 1.6
     */
    public static FileObject[] getChildren(FileObject directory, FileObject[] fallback) {
        try {
            return directory.getChildren();
        } catch (Exception ignored) {
        }
        return fallback;
    }

    /**
     * Checks if the given URL is a root directory.
     *
     * @param url The directory
     * @return Root?
     * @since 1.6
     */
    public static boolean isRoot(FileObject url) {
        return VFSUtils.isDirectory(url, false) && VFSUtils.getParent(url) == null;
    }

    /**
     * Tries to remove all directories and files from a given directory.
     * <p>
     * Use this method with care since all entries are removed recursively without transferring
     * them into the Windows system garbage or so!
     *
     * @param target The directory to be cleared
     * @throws NullPointerException If parameters are <code>null</code>
     * @since 1.6
     */
    public static void clearDirectory(final FileObject target) {

        if (target == null) throw new NullPointerException();

        // Never call from EDT since directory listing may block the GUI!
        if (SwingUtilities.isEventDispatchThread()) {
            new Thread(() -> clearDirectory(target), VFSLibSettings.LOG_PREFIX + "VFS Utils Cleaner").start();
            return;
        }

        // First we have to determine the directory entries
        // including all sub-directories and files.

        Vector<FileObject> entries = new Vector<>(0);
        VFSUtils.collectEntries(target, entries);

        // Then let's remove the entries in the reverse order
        // since files must be deleted before the directories.
        int size = entries.size();
        for (int i = (size - 1); i >= 0; i--) {
            try {
                entries.elementAt(i).delete();
            } catch (Exception ignored) {
            }
        }
        entries.removeAllElements();
    }

    /**
     * Searches the given directory recursively for files and directories adds the results to the
     * container.
     * <p>
     * The search directory itself is NOT included!
     *
     * @param searchdir The directory to begin searching
     * @param entries   Container with paths (<code>File</code>)
     * @since 1.6
     */
    public static void collectEntries(FileObject searchdir, Vector<FileObject> entries) {

        if (searchdir == null || entries == null) return;

        // Let's begin with the parental directory
        if (!VFSUtils.isDirectory(searchdir, false)) return;

        try {
            // Are there any appropriate files?
            FileObject[] urls = VFSUtils.getChildren(searchdir);  // Warns for EDT
            for (FileObject url : urls) {

                entries.addElement(url);
                if (VFSUtils.isDirectory(url, false)) {

                    // We have to perform recursion here
                    VFSUtils.collectEntries(url, entries);
                }
            }
        } catch (Exception ignored) {
        }
    }

    /**
     * Convenience to check if a given URL exists and is a regular file.
     *
     * @param url The URL to be checked
     * @return File?
     * @throws NullPointerException If the URL is <code>null</code>
     * @since 1.6
     */
    public static boolean isExistingFile(FileObject url) {
        return VFSUtils.exists(url, false) && VFSUtils.isFile(url, false);
    }

    /**
     * Writes binary data into a network file.
     *
     * @param url     Network file to be written
     * @param istream Binary data stream
     * @param length  Number of bytes to write, -1 = unknown
     * @return Successful?
     * @since 1.6
     */
    public static boolean writeFile(FileObject url, InputStream istream, long length) {

        byte[] buffer = new byte[1024];
        int len;

        try {
            if (length != -1) {
                try {
                    // Allow proper upload for cloud file systems, "Content-Length" must be set
                    if (VFSUtils.isContentLengthRequired(url)) {
                        url.getContent().setAttribute(VFSLibConstants.ATTR_CONTENT_LENGTH, length);
                    }
                } catch (Exception ignored) {
                }
            }

            OutputStream ostream = url.getContent().getOutputStream();
            while ((len = istream.read(buffer)) != -1) {
                ostream.write(buffer, 0, len);
            }
            try {
                ostream.flush();
            } catch (Exception ignored) {
            }
            ostream.close();
            return true;
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Checks, if the given container holds the given URL.
     * <p>
     * This is a workaround since <code>FileObject.equals()</code> does not properly detect
     * identical entries properly. Here, the text representations are checked instead.
     *
     * @param container The URL set
     * @param url       The URL to be checked
     * @return URL is element of container?
     * @throws NullPointerException If the file is <code>null</code>
     * @since 1.6
     */
    public static boolean contains(Vector<FileObject> container, FileObject url) {

        try {
            for (FileObject next : container) {
                if (String.valueOf(next).equals(String.valueOf(url))) return true;
            }
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Tries to delete the given network file/directory (directory must be empty).
     *
     * @param target The directory to be removed
     * @return Successful?
     * @throws NullPointerException If parameters are <code>null</code>
     * @since 1.6
     */
    public static boolean delete(FileObject target) {

        if (target == null) throw new NullPointerException();

        try {
            return target.delete();
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Tries to delete the given network directory recursively.
     *
     * @param target The directory to be removed
     * @throws NullPointerException If parameters are <code>null</code>
     * @since 2.8
     */
    public static void deleteAll(FileObject target) {

        if (target == null) throw new NullPointerException();

        try {
            target.deleteAll();
        } catch (Exception ignored) {
        }
    }

    /**
     * Moves the given network file into the archive subfolder.
     *
     * @param source    File to be moved
     * @param subfolder Defaults to "ARCHIVE"
     * @return Successful?
     * @throws NullPointerException If parameters are <code>null</code>
     * @since 1.6
     */
    public static boolean moveToArchive(FileObject source, String subfolder) {

        if (source == null) throw new NullPointerException();
        if (subfolder == null) subfolder = "ARCHIVE";

        try {
            // Move file to subdirectory, do not overwrite!
            String path = source.getParent().toString() + '/' + subfolder + '/' + source.getName().getBaseName();
            VFSFileSystemManager vfsmanager = VFSLibSettings.getManagerNetwork();
            FileObject target = vfsmanager.resolveFile(path);
            if (target.exists()) {
                VFSLibSettings.log(Level.CONFIG, "moveToArchive(): target=" + VFSUtils.getDisplayURIUsernameOnly(target, null) + " exists (not overwritten)");
            } else {
                VFSLibSettings.log(Level.CONFIG, "moveToArchive(): target=" + VFSUtils.getDisplayURIUsernameOnly(target, null));
                if (!target.exists()) target.createFile();
                if (!target.exists()) return false;  // Permissions?

                VFSLibSettings.log(Level.CONFIG, "moveToArchive(): move " + VFSUtils.getDisplayURIUsernameOnly(source, null) + " to " + VFSUtils.getDisplayURIUsernameOnly(target, null));
                source.moveTo(target);
            }
            return true;
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Any file system specific options available (internal cache not empty)?
     *
     * @param options The options to check
     * @return Options available?
     * @since 1.6
     */
    public static boolean hasFileSystemOptions(FileSystemOptions options) {
        try {
            // Get the internal cache via reflection
            Map map = (Map) ReflectionUtils.getDeclaredField(options, "options");
            return !map.isEmpty();
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Prints file system specific options for a connection.
     *
     * @param options The options to check
     * @return Text
     * @since 1.6
     */
    public static String debugPrint(FileSystemOptions options) {

        StringBuilder builder = new StringBuilder(0);
        try {
            // Get the internal cache via reflection
            // private Map<FileSystemOptionKey, Object> options = new TreeMap<FileSystemOptionKey, Object>();
            Map map = (Map) ReflectionUtils.getDeclaredField(options, "options");

            String clazz = options.getClass().getName();
            builder.append("\nTotal variable count (" + clazz + "): " + map.size() + "\n\n");

            Vector collect = new Vector(0);
            Iterator iterator = map.keySet().iterator();
            while (iterator.hasNext()) {

                Object key = iterator.next();  // FileSystemOptionKey (internal class wraps String)
                collect.addElement(key);
            }
            Comparator comparator = (o1, o2) -> {
                try {
                    String name1 = (String) ReflectionUtils.getDeclaredField(o1, "name");
                    String name2 = (String) ReflectionUtils.getDeclaredField(o2, "name");
                    return name1.compareTo(name2);
                } catch (Exception ignored) {
                }
                return 0;
            };
            Collections.sort(collect, comparator);

            int size = collect.size();
            for (int i = 0; i < size; i++) {
                Object key = collect.elementAt(i);
                String name = (String) ReflectionUtils.getDeclaredField(key, "name");

                Object value = map.get(key);
                builder.append(name).append('=').append(value).append('\n');
            }
            collect.removeAllElements();

            builder.append('\n');
        } catch (Exception ignored) {
        }
        return builder.toString();
    }

    /**
     * Determines the VFS connection associated with the given ID.
     *
     * @param vfsmanager The manager
     * @param id         The connection ID
     * @return The connection, <code>null</code> if not available
     * @since 1.6
     */
    public static VFSConnection findVFSConnection(VFSFileSystemManager vfsmanager, String id) {

        for (VFSConnection vfsconn : vfsmanager.getConnections()) {
            if (vfsconn.getID("").equals(id)) return vfsconn;
        }
        return null;
    }

    /**
     * Determines the VFS connections which can be used to show the given network file.
     * <p>
     * The connection must be available (scheme supported) and applicable for the given URL
     * (host, credentials etc. must match).
     *
     * @param vfsmanager The manager
     * @param entry      The URL
     * @return The connections, <code>null</code> if not available
     * @since 1.6
     */
    public static VFSConnection[] findVFSConnections(VFSFileSystemManager vfsmanager, FileObject entry) {

        VFSConnection[] vfsconns = new VFSConnection[0];
        Vector<VFSConnection> connections = vfsmanager.getConnections();

        // Prio 1: List connections first which include the user info
        for (VFSConnection vfsconn : connections) {
            if (vfsconn.isAvailable() && vfsconn.isApplicable(String.valueOf(entry), true)) {  // Check user
                vfsconns = (VFSConnection[]) JavaUtils.addToArray(vfsconns, vfsconn);
            }
        }

        // Prio 2: List those connections without or with other user for the system
        for (VFSConnection vfsconn : connections) {                              // Ignore user
            if (vfsconn.isAvailable() && vfsconn.isApplicable(String.valueOf(entry), false) && !JavaUtils.elementOf(vfsconn, vfsconns)) {
                vfsconns = (VFSConnection[]) JavaUtils.addToArray(vfsconns, vfsconn);
            }
        }
        return vfsconns;
    }

    /**
     * Determines the VFS connections which can be used to show the given network file.
     * <p>
     * The connection must be available (scheme supported) and applicable for the given URL
     * (host, credentials etc. must match).
     *
     * @param vfsmanager The manager
     * @param url        The URL
     * @return The connections, <code>null</code> if not available
     * @since 1.6
     */
    public static VFSConnection[] findVFSConnections(VFSFileSystemManager vfsmanager, String url) {

        VFSConnection[] vfsconns = new VFSConnection[0];
        Vector<VFSConnection> connections = vfsmanager.getConnections();

        // Prio 1: List connections first which include the user info
        for (VFSConnection vfsconn : connections) {
            if (vfsconn.isAvailable() && vfsconn.isApplicable(url, true)) {  // Check user
                vfsconns = (VFSConnection[]) JavaUtils.addToArray(vfsconns, vfsconn);
            }
        }

        // Prio 2: List those connections without or with other user for the system
        for (VFSConnection vfsconn : connections) {            // Ignore user
            if (vfsconn.isAvailable() && vfsconn.isApplicable(url, false) && !JavaUtils.elementOf(vfsconn, vfsconns)) {
                vfsconns = (VFSConnection[]) JavaUtils.addToArray(vfsconns, vfsconn);
            }
        }
        return vfsconns;
    }

    /**
     * Convenience to automatically apply user and password defined for a connection.
     *
     * @param uri     The URI to resolve
     * @param vfsconn The connection with options
     * @return The modified URI, <code>uri</code> parameter otherwise
     * @since 1.6
     */
    public static String addMissingCredentials(String uri, VFSConnection vfsconn) {

        if (vfsconn != null) {
            try {
                // Connection has user and password?
                String connuser = vfsconn.getUser(null);
                String connpwd = vfsconn.getPassword(null);
                if (connuser != null && !connuser.isEmpty() && connpwd != null && !connpwd.isEmpty()) {

                    // User/password are missing in given URI?
                    // Caution: Do not use URI here, since included SPACEs etc. lead to exception (OK for VFS)!
                    // Caution: Do not use URL here, since "sftp" is not a supported protocol (exception)!
                    VFSURI uriobj = new VFSURI(uri);
                    String uriuser = uriobj.getUserInfo();  // aschwolow:[pwd]
                    if (uriuser == null || uriuser.isEmpty() || uriuser.indexOf(':') == -1) {  // No password
                        uriobj = new VFSURI(uriobj.getScheme(),
                                connuser + ':' + connpwd,  // Replace/insert
                                uriobj.getHost(),
                                uriobj.getPort(),
                                uriobj.getPath(),
                                uriobj.getQuery(),
                                uriobj.getFragment());
                        return String.valueOf(uriobj);
                    }
                }
            } catch (Exception ignored) {
            }
        }
        return uri;  // Failsafe
    }

    /**
     * Checks if the given network files are on the same server.
     * <p>
     * Should be checked when the <code>FileObject.moveTo()</code> feature is to be used.
     *
     * @param url1 The first  network file
     * @param url2 The second network file
     * @return The modified URI, <code>uri</code> parameter otherwise
     * @since 1.6
     */
    public static boolean isSameServer(FileObject url1, FileObject url2) {

        try {
            // Normalize:  sftp://mycompany.com:666
            VFSURI temp = new VFSURI(url1.toString());
            String raw1 = new VFSURI(temp.getScheme(),
                    null,
                    temp.getHost(),
                    temp.getPort(),
                    null,  // No path here!
                    null, null).toString();

            temp = new VFSURI(url2.toString());
            String raw2 = new VFSURI(temp.getScheme(),
                    null,
                    temp.getHost(),
                    temp.getPort(),
                    null,  // No path here!
                    null, null).toString();
            return raw1.equals(raw2);
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Is the given VFS protocol/scheme stateful?
     * <p>
     * Stateful measn that a connection requires a session to be established and closed.
     * For example, FTP and SFTP are stateful protocols, while HTTP is not.
     *
     * @param protocol The VFS scheme/protocol like "sftp"
     * @return Stateful?
     * @since 1.6
     */
    public static boolean isStateful(String protocol) {

        return protocol.equals("ftp") || protocol.equals("ftps") ||
                protocol.equals("sftp") || protocol.equals("webdav") ||
                protocol.equals(VFSLib.getSharedInstance().getSchemeDropbox());
    }

    /**
     * Extracts the scheme part of the given URI.
     *
     * @param uri The URI to parse
     * @return The protocol like "sftp", <code>null</code> otherwise
     * @since 1.6
     */
    public static String getProtocol(String uri) {
        try {
            // Caution: Do not use URI here, since included SPACEs etc. lead to exception (OK for VFS)!
            // Caution: Do not use URL here, since "sftp" is not a supported protocol (exception)!
            return new VFSURI(uri).getScheme();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Provides the numerical port number appropriate for the given VFS protocol/scheme.
     *
     * @param protocol The VFS scheme/protocol like "sftp"
     * @return The port, -1 otherwise (e.g. for "zip")
     * @since 1.6
     */
    public static int getDefaultPort(String protocol) {

        switch (protocol) {
            case "ftp":
                return 21;  // ftp-data=20, ftp=21
            case "ftps":
                return 21;  // Normal FTP, send AUTH TLS, then encrypted
            case "http":
            case "webdav":
                return 80;
            case "https":
                return 443;
            case "sftp":
                return 22;
            default:
                return -1;
        }
    }

    /**
     * Creates a new comparator used together with <code>TreeMap</code> to handle <code>FileObject</code>'s properly.
     * <p>
     * Caution: Do not use Hashtable<FileObject, ... since FileObject's for the same folder
     * use different hash codes. Such folders are then added multiple times and not once as expected!
     *
     * @return The comparator
     * @since 1.6
     */
    public static Comparator<FileObject> getFileObjectComparator() {

        // Caution: Do not use Hashtable<FileObject, ... here since FileObject's for the same folder
        // use different hash codes. Such folders are then added multiple times and not once as expected!

        return Comparator.comparing(String::valueOf);
    }

    /**
     * Checks if the given protocol requires setting the <code>VFSLibConstants.ATTR_CONTENT_LENGTH</code> attribute.
     * <p>
     * This attribute is known to be required for proper file handling with the Amazon S3 provider.
     *
     * @param url The network file
     * @return Required?
     * @throws NullPointerException If a parameter is <code>null</code>
     * @since 1.6
     */
    public static boolean isContentLengthRequired(FileObject url) {

        if (url == null) throw new NullPointerException();

        try {
            // Caution: Do not use URI here, since included SPACEs etc. lead to exception (OK for VFS)!
            // Caution: Do not use URL here, since "sftp" is not a supported protocol (exception)!
            String scheme = new VFSURI(String.valueOf(url)).getScheme();

            // Only S3 so far, MUST not be set for SFTP due to OOME for large files (see OTRS 2014060511000013)!
            return scheme.equals(VFSLib.getSharedInstance().getSchemeAmazonS3());
        } catch (Exception ignored) {
        }
        return false;
    }

    /**
     * Checks if the given network file contents is equal to the given local file.
     *
     * @param url  The network file
     * @param file The local file
     * @return Contents match?
     * @throws NullPointerException If a parameter is <code>null</code>
     * @since 2.8
     */
    public static boolean compareContents(FileObject url, File file) {

        byte[] bufferurl = new byte[1024];
        byte[] bufferfile = new byte[1024];
        int lenurl;
        int lenfile;

        try (InputStream instreamurl = url.getContent().getInputStream(); // try-with-resources
             InputStream instreamfile = Files.newInputStream(file.toPath())) {

            while ((lenurl = instreamurl.read(bufferurl)) != -1) {
                lenfile = instreamfile.read(bufferfile);
                if (lenurl != lenfile) return false;
                if (!Arrays.equals(bufferurl, bufferfile)) return false;
            }
            return true;
        } catch (Exception ignored) {
        }
        return false;
    }
}
