/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.net;

import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.lang.ReflectionUtils;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.impl.DefaultFileSystemManager;
import org.apache.commons.vfs2.impl.StandardFileSystemManager;
import org.apache.commons.vfs2.operations.FileOperationProvider;
import org.apache.commons.vfs2.provider.FileProvider;
import org.apache.commons.vfs2.provider.LocalFileProvider;

import javax.swing.*;
import java.io.File;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.Map;
import java.util.Vector;
import java.util.logging.Level;


/**
 * Commons specific <code>FileSystemManager</code> managing protocol options etc.
 * <p>
 * Main reason for this class is the possiblity to install options for each of the schemes
 * which are used by the overwritten <code>resolve()</code> methods automatically. The original
 * <code>FileSystemManager</code> implementations also lack support for various functions
 * as to mention the direct removal of already installed schemes.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class VFSFileSystemManager extends StandardFileSystemManager {


    /**
     * Cache for the currently configured VFS connections.
     */
    protected Vector<VFSConnection> connections;


    /**
     * Constructor method using the default configuration.
     *
     * @since 1.6
     */
    public VFSFileSystemManager() {
        this(StandardFileSystemManager.class.getResource("providers.xml"));  // Propagate
    }

    /**
     * Constructor method using the given configuration.
     *
     * @param configuri The descriptor resource path
     * @throws NullPointerException If a parameter is <code>null</code>
     * @since 1.6
     */
    public VFSFileSystemManager(URL configuri) {

        if (configuri == null) throw new NullPointerException();

        this.connections = new Vector<>(0);

        try {
            // Do the same as VFS.getManager() does
            Class clazz = Class.forName("org.apache.commons.vfs2.impl.StandardFileSystemManager");
            super.setClassLoader(clazz.getClassLoader());  // Do not use our Commons class loader
            super.setConfiguration(configuri);
            super.init();

            // The scheme specific options are set by the VFSConnection objects for each resolve
        } catch (final Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
    }

    /**
     * Cares for proper cleanup after releasing the object.
     *
     * @since 1.6
     */
    @Override
    protected void finalize() throws Throwable {

        super.finalize();

        if (this.connections != null) this.connections.removeAllElements();

        try {
            super.close();
        } catch (Exception ignored) {
        }
    }

    /**
     * Provides the cache with scheme providers.
     *
     * @return The cache, may be <code>null</code>
     * @since 1.6
     */
    public Map getProviders() {
        try {
            return (Map) ReflectionUtils.getDeclaredField(this, DefaultFileSystemManager.class, "providers");
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Provides the provider for the given scheme/protocol.
     *
     * @param scheme The protocol like "sftp"
     * @return The provider, may be <code>null</code>
     * @since 1.6
     */
    public FileProvider getProvider(String scheme) {
        try {
            return (FileProvider) getProviders().get(scheme);
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Removes the <code>FileOperationProvider</code> for the specified scheme.
     * <p>
     * Another alternative would be to specify a modified "providers.xml" for the 2nd constructor.
     * The original <code>FileSystemManager</code> lacks this functionality of directly removing
     * already installed providers.
     *
     * @param protocol The VFS scheme like "sftp"
     * @return The provider, may be <code>null</code>
     * @since 1.6
     */
    public FileOperationProvider removeOperationProvider(String protocol) {
        try {
            return (FileOperationProvider) this.getProviders().remove(protocol);
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Locates a file by URI.
     * <p>
     * If installed the file system options e.g. for schemes like "sftp" are applied here.
     *
     * @param uri The URI to resolve
     * @return The network file
     * @throws FileSystemException If an error occurs
     * @since 1.6
     */
    @Override
    public FileObject resolveFile(String uri) throws FileSystemException {

        // Method warns if EDT
        return resolveFile(uri, (VFSConnection) null);  // Makes best guess
    }

    /**
     * Locates a file by URI.
     * <p>
     * If installed the file system options e.g. for schemes like "sftp" are applied here.
     *
     * @param uri     The URI to resolve
     * @param vfsconn The connection with options
     * @return The network file
     * @throws FileSystemException If an error occurs
     * @since 1.6
     */
    public FileObject resolveFile(String uri, VFSConnection vfsconn) throws FileSystemException {

        if (!uri.startsWith("file:/") && SwingUtilities.isEventDispatchThread()) {
            VFSLibSettings.log(Level.WARNING, new Throwable("Network files must not be resolved from EDT!"));
        }

        FileSystemOptions options = (vfsconn != null ? vfsconn.getFileSystemOptions() : getBestFileSystemOptions(uri));
        uri = VFSUtils.addMissingCredentials(uri, (vfsconn != null ? vfsconn : getBestConnection(uri)));
        return super.resolveFile(uri, options);
    }

    /**
     * Locates a file by URI and file system options.
     * <p>
     * If installed the file system options e.g. for schemes like "sftp" are applied here.
     *
     * @param uri     The URI to resolve
     * @param options The file system options
     * @return The network file
     * @throws FileSystemException If an error occurs
     * @since 1.6
     */
    @Override
    public FileObject resolveFile(String uri, FileSystemOptions options) throws FileSystemException {

        // Method warns if executed by EDT
        return resolveFile(uri, options, null);  // Makes best guess
    }

    /**
     * Locates a file by URI.
     * <p>
     * If installed the file system options e.g. for schemes like "sftp" are applied here.
     *
     * @param uri     The URI to resolve
     * @param options The file system options
     * @param vfsconn The connection with options
     * @return The network file
     * @throws FileSystemException If an error occurs
     * @since 1.6
     */
    public FileObject resolveFile(String uri, FileSystemOptions options, VFSConnection vfsconn) throws FileSystemException {

        if (!uri.startsWith("file:/") && SwingUtilities.isEventDispatchThread()) {
            VFSLibSettings.log(Level.WARNING, new Throwable("Network files must not be resolved from EDT!"));
        }
        uri = VFSUtils.addMissingCredentials(uri, (vfsconn != null ? vfsconn : getBestConnection(uri)));
        return super.resolveFile(uri, options);
    }

    /**
     * Resolves a URI, relative to a base file.
     * <p>
     * If installed the file system options e.g. for schemes like "sftp" are applied here.
     *
     * @param baseFile The base file
     * @param uri      The URI to resolve
     * @return The network file
     * @throws FileSystemException If an error occurs
     * @since 1.6
     */
    @Override
    public FileObject resolveFile(File baseFile, String uri) throws FileSystemException {
        return resolveFile(baseFile, uri, null);  // Makes best guess
    }

    /**
     * Resolves a URI, relative to a base file.
     * <p>
     * If installed the file system options e.g. for schemes like "sftp" are applied here.
     *
     * @param baseFile The base file
     * @param uri      The URI to resolve
     * @param vfsconn  The connection with options
     * @return The network file
     * @throws FileSystemException If an error occurs
     * @since 1.6
     */
    public FileObject resolveFile(File baseFile, String uri, VFSConnection vfsconn) throws FileSystemException {

        // TODO: Test.
        FileSystemOptions options = (vfsconn != null ? vfsconn.getFileSystemOptions() : getBestFileSystemOptions(uri));
        uri = VFSUtils.addMissingCredentials(uri, (vfsconn != null ? vfsconn : getBestConnection(uri)));

        try {
            // Call private method via reflection
            Method method_getLocalFileProvider = DefaultFileSystemManager.class.getDeclaredMethod("getLocalFileProvider");
            method_getLocalFileProvider.setAccessible(true);
            LocalFileProvider provider = (LocalFileProvider) method_getLocalFileProvider.invoke(this);
            return super.resolveFile(provider.findLocalFile(baseFile), uri, options);
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
            throw new FileSystemException(e);
        }
    }

    /**
     * Resolves a URI, relative to a base file.
     * <p>
     * If installed the file system options e.g. for schemes like "sftp" are applied here.
     *
     * @param baseFile The base file
     * @param uri      The URI to resolve
     * @return The network file
     * @throws FileSystemException If an error occurs
     * @since 1.6
     */
    @Override
    public FileObject resolveFile(FileObject baseFile, String uri) throws FileSystemException {
        return resolveFile(baseFile, uri, (VFSConnection) null);  // Makes best guess
    }

    /**
     * Resolves a URI, relative to a base file.
     * <p>
     * If installed the file system options e.g. for schemes like "sftp" are applied here.
     *
     * @param baseFile The base file
     * @param uri      The URI to resolve
     * @param vfsconn  The connection with options
     * @return The network file
     * @throws FileSystemException If an error occurs
     * @since 1.6
     */
    public FileObject resolveFile(FileObject baseFile, String uri, VFSConnection vfsconn) throws FileSystemException {

        if (!String.valueOf(baseFile).startsWith("file:/") && SwingUtilities.isEventDispatchThread()) {
            VFSLibSettings.log(Level.WARNING, new Throwable("Network files must not be resolved from EDT!"));
        }

        // TODO: Test.
        FileSystemOptions options = (vfsconn != null ? vfsconn.getFileSystemOptions() : getBestFileSystemOptions(uri));
        uri = VFSUtils.addMissingCredentials(uri, (vfsconn != null ? vfsconn : getBestConnection(uri)));
        return super.resolveFile(baseFile, uri, options);
    }

    /**
     * Provides the currently configured network connections for this manager.
     *
     * @return The connections (original reference, take care)
     * @since 1.6
     */
    public Vector<VFSConnection> getConnections() {
        return this.connections;
    }

    /**
     * Sets the currently configured network connections for this manager.
     *
     * @param connections The connections
     * @throws NullPointerException If a parameter is <code>null</code>
     * @since 1.6
     */
    public void setConnections(Vector<VFSConnection> connections) {

        if (connections == null) throw new NullPointerException();

        this.connections = connections;
    }

    /**
     * Adds a new network connections for this manager.
     *
     * @param vfsconn The connection
     * @throws NullPointerException If a parameter is <code>null</code>
     * @since 1.6
     */
    public void addConnection(VFSConnection vfsconn) {

        if (vfsconn == null) throw new NullPointerException();

        this.connections.addElement(vfsconn);
    }

    /**
     * Searches the configured connections for an appropriate instance (scheme, host, ...).
     * <p>
     * This way it is possible to access files on servers with special params like passive FTP by a simple
     * URL which may be specified on the command line (and which lacks the connection options).
     *
     * @param uri The network file to check
     * @return The connection, may be <code>null</code>
     * @since 1.6
     */
    public VFSConnection getBestConnection(String uri) {

        VFSConnection[] candidates = VFSUtils.findVFSConnections(this, uri);
        VFSConnection conn = null;
        if (candidates.length >= 1) {
            String mapped = VFSUtils.getDisplayURIUsernameOnly(uri, candidates[0]);
            VFSLibSettings.log(Level.FINE, "Using first-best configured connection " + candidates[0].getName("?") + " for uri=" + mapped);
            conn = candidates[0];  // Take first-best
        }
        return conn;
    }

    /**
     * Searches the configured connections for an appropriate instance (scheme, host, ...) and provides its options.
     * <p>
     * This way it is possible to access files on servers with special params like passive FTP by a simple
     * URL which may be specified on the command line (and which lacks the connection options).
     *
     * @param uri The network file to check
     * @return The options, may be <code>null</code>
     * @since 1.6
     */
    protected FileSystemOptions getBestFileSystemOptions(String uri) {

        VFSConnection[] candidates = VFSUtils.findVFSConnections(this, uri);
        FileSystemOptions options = null;
        if (candidates.length >= 1) {
            String mapped = VFSUtils.getDisplayURIUsernameOnly(uri, candidates[0]);
            VFSLibSettings.log(Level.FINE, "Using first-best configured connection options " + candidates[0].getName("?") + " for uri=" + mapped);
            options = candidates[0].getFileSystemOptions();  // Take first-best
        } else {
            try {
                // Failsafe: Use preconfigured options e.g. do not perform strict key checking for SFTP
                // Caution: Do not use URI here, since included SPACEs etc. lead to exception (OK for VFS)!
                // Caution: Do not use URL here, since "sftp" is not a supported protocol (exception)!
                String scheme = new VFSURI(uri).getScheme();
                options = VFSConnection.getDefaultFileSystemOptions(scheme, null);
            } catch (Exception ignored) {
            }
        }
        return options;
    }
}