/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.net;

import java.util.Hashtable;
import java.util.Vector;


/**
 * Handles multiple network connections to be recycled.
 * <p>
 * Standard VFS lacks support for connection pooling. For every resolve a new connection over FTP, SFTP etc.
 * is established. This is OK for single files being resolved but not suitable for FTP client applications.
 * If a FTP or SFTP connection has been established once it is cached and re-used whenever an appropriate
 * network file is being resolved.
 * <p>
 * This pool mechanism is appropriate for stateful protocols like FTP, FTPS, and SFTP (not for HTTP).
 * <p>
 * Helpful commands to monitor incoming FTP/SFTP connections under Linux:
 * <code>
 * while (true) ; do clear;ps aux|grep vsftp ; sleep 1; done
 * while (true) ; do clear;ps aux|grep sshd: ; sleep 1; done
 * </code>
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class ClientPool /*extends ManagedFactory*/ { // TODO


    /**
     * Shared <code>ClientPool</code> instance.
     */
    protected static ClientPool sharedInstance;
    /**
     * Recycled strategy instance.
     */
    //protected static TimeoutStrategy STRATEGY = new TimeoutStrategy(10);  // 10 minutes

    // The cacheObjectByID cache of the base class is not used here!

    /**
     * Remembers all clients currently managed by this factory, managed object is <code>ClientWrapper</code>.
     */
    //protected Hashtable<String, Vector<ManagedObject>> cachePool = null;
    /**
     * Remembers clients from the currently occupied.
     */
    protected final Vector<ClientWrapper> cacheLocked;
    /**
     * Logs the requesters of the clients in pool (key: client, value: call stack).
     */
    protected final Hashtable<ClientWrapper, StackTraceElement[]> cacheLockedBy;


    /**
     * Default constructor.
     *
     * @since 1.6
     */
    public ClientPool() {

        // Allow for automatic release by applications, cacheObjectByID is not used here
        //FactoryManager.registerFactory(this);

        //this.cachePool = new Hashtable<>(0);
        this.cacheLocked = new Vector<>(0);
        this.cacheLockedBy = new Hashtable<>(0);
    }

    /**
     * Clean-up method to help the gc.
     *
     * @throws Throwable Error indication
     * @since 1.6
     */
    @Override
    protected void finalize() throws Throwable {

        super.finalize();
/*
        Enumeration<String> en = this.cachePool.keys();
        while (en.hasMoreElements()) {
            Vector<ManagedObject> pool = this.cachePool.get(en.nextElement());

            int size = pool.size();
            for (int i = 0; i < size; i++) {
                ClientWrapper client = (ClientWrapper) pool.elementAt(i).object;  // Close connections properly
                client.disconnectClient();
            }
            pool.removeAllElements();
        }
        this.cachePool.clear();

        this.cacheLocked.removeAllElements();
        this.cacheLockedBy.clear();
 */
    }

    /**
     * Provides the shared instance of the factory implementation.
     *
     * @return Shared instance of the factory
     * @since 1.6
     */
    public static synchronized ClientPool getSharedInstance() {

        if (sharedInstance == null) sharedInstance = new ClientPool();
        return sharedInstance;
    }

    /**
     * Provides an open client for the network, recycled from the pool (same as <code>getFreeClient()</code>).
     * <p>
     * Here, a free network client can be requested. If none is available the method returns <code>null</code>.
     * The provided network client is locked here automatically and must be unlocked explicitely to be recycled.
     * If it returns <code>null</code> you have to create a new client and set it here using <code>addClient()</code>.
     * This is necessary since we do not have required parameters like file system options here.
     *
     * @param id The factory-unique identifier of the shared object (<code>String</code> here, unique URL)
     * @return The shared object, <code>null</code> if none could be created (<code>ClientWrapper</code> here)
     * @see #getFreeClient(String)
     * @since 1.6
     */
//    @Override
    public synchronized Object request(Object id) {

        String url = id instanceof String ? (String) id : String.valueOf(id);
/*
        // If possible, return an existing instance from the pool
        Vector<ManagedObject> pool = this.cachePool.get(url);
        if (pool != null) {

            // Search for free client in existing pool
            int size = pool.size();
            for (int i = 0; i < size; i++) {
                ManagedObject managed = pool.elementAt(i);
                ClientWrapper client = (ClientWrapper) managed.object;

                if (!this.cacheLocked.contains(client)) {  // Available!
                    VFSLibSettings.log(Level.CONFIG, "Recycled client " + url + " (" + Integer.toHexString(client.hashCode()) + ") from pool");
                    //Thread.dumpStack();
                    //debugPrint();

                    // Log who requested the client (debugging)
                    this.cacheLocked.addElement(client);
                    this.cacheLockedBy.put(client, Thread.currentThread().getStackTrace());

                    // Update some strategy relevant fields and return instance
                    managed.lastRequest = new Date();
                    managed.requestCount++;
                    return client;
                }
            }
        } else {
            // Pool does not exist, let's create it
            pool = new Vector<ManagedObject>(0);
            this.cachePool.put(url, pool);
        }
 */
        return null;
        // No free connection available, must be create outside, then set here using addClient().
        // This is necessary since we do not have required parameters like file system options here.
    }

    /**
     * Provides an open client for the network, recycled from the pool if possible (convenience).
     *
     * @param url The client URL including login, port etc.
     * @return Open client, <code>null</code> if an error occurred
     * @throws NullPointerException If parameters are <code>null</code>
     * @since 1.6
     */
    public ClientWrapper getFreeClient(String url) {

        return (ClientWrapper) ClientPool.getSharedInstance().request(url);
    }

    /**
     * Adds a new client for the pool, normally added by the provider client factories.
     * <p>
     * The new client is locked automatically here.
     *
     * @param client Client to be added
     * @throws NullPointerException If parameters are <code>null</code>
     * @since 1.6
     */
    public synchronized void addClient(ClientWrapper client) {
/*
        try {
            String url = client.getURLID();
            Vector<ManagedObject> pool = this.cachePool.get(url);
            if (pool == null) {
                // Pool does not exist, let's create it
                pool = new Vector<ManagedObject>(0);
                this.cachePool.put(url, pool);
            }

            // Create new managed object for the client
            String clazz = client.getClass().getName();
            ManagedObject managed = new ManagedObject(client, url, clazz, STRATEGY, new Date(), 0, 0);
            pool.addElement(managed);

            // Log who requested the client (debugging)
            this.cacheLocked.addElement(client);
            this.cacheLockedBy.put(client, Thread.currentThread().getStackTrace());

            // Update some strategy relevant fields and return instance
            managed.lastRequest = new Date();
            managed.requestCount++;
            VFSLibSettings.log(Level.CONFIG, "Added new client " + url + " (" + Integer.toHexString(client.hashCode()) + ") to pool");
            //debugPrint();
        } catch (Exception e) {
            //e.printStackTrace();
        }

 */
    }

    /**
     * Releases an occupied client so that it can be re-used.
     *
     * @param client Client to be released
     * @return Released successfully?
     * @throws NullPointerException If parameters are <code>null</code>
     * @since 1.6
     */
    public synchronized boolean unlockClient(ClientWrapper client) {

        if (client == null) return false;
/*
        try {
            ManagedObject managed = findManagedObject(client);  // Managed here at all?
            if (managed != null && this.cacheLocked.contains(client)) {  // Must be locked

                // Client remains in the pool!
                this.cacheLocked.removeElement(client);
                this.cacheLockedBy.remove(client);

                // Locked time does not count, released after a time period of inactivity
                managed.lastRequest = new Date();
                VFSLibSettings.log(Level.CONFIG, "Unlocked client " + client.toString() + " (" + Integer.toHexString(client.hashCode()) + ") in pool");
                return true;
            }
        } catch (Exception e) {
            //VFSLibSettings.log(Level.WARNING, e);
        }

 */
        return false;
    }

    /**
     * Searches the client container for the associated managed object.
     *
     * @param client The client
     * @return The managed object for the client, may be <code>null</code>
     * @since 1.6
     */
    protected /*ManagedObject*/void findManagedObject(ClientWrapper client) {
/*
        Enumeration<String> keys = this.cachePool.keys();
        while (keys.hasMoreElements()) {
            String uri = keys.nextElement();
            Vector<ManagedObject> pool = this.cachePool.get(uri);

            for (ManagedObject managed : pool) {
                ClientWrapper nextclient = (ClientWrapper) managed.object;
                if (nextclient == client) return managed;
            }
        }
        return null;
 */
    }

    /**
     * Checks whether a factory shared object can be released or not.
     *
     * @param id The factory-unique identifier of the shared object
     * @return Can the object be released according to the factories needs?
     * @since 1.6
     */
    //@Override
    public synchronized boolean canRelease(Object id) {
        // Release strategy is the master, see releaseByStrategy() and releaseForced()
        return true;
    }

    /**
     * Releases the given shared objects immediately (free clients here).
     *
     * @param id The factory-unique identifier of the shared object
     * @throws NullPointerException If a parameter is <code>null</code>
     * @since 1.6
     */
    //@Override
    public synchronized void release(Object id) {

        String url = id instanceof String ? (String) id : String.valueOf(id);
/*
        Vector<ManagedObject> pool = this.cachePool.get(url);
        if (pool != null) {

            // Release only free clients in the pool!
            Vector<ManagedObject> toremove = new Vector<ManagedObject>(0);
            for (ManagedObject managed : pool) {
                ClientWrapper client = (ClientWrapper) managed.object;
                if (!this.cacheLocked.contains(client)) toremove.addElement(managed);
            }
            for (ManagedObject managed : toremove) {  // Avoid exceptions

                // Disconnect from server first
                ClientWrapper client = (ClientWrapper) managed.object;
                client.disconnectClient();
                pool.remove(managed);
                VFSLibSettings.log(Level.CONFIG, "Released object " + client.toString());
            }
            toremove.removeAllElements();
        }
 */
    }

    /**
     * Releases all managed shared objects immediately (free clients here).
     * <p>
     * This functionality can be used in order to make as much of occupied memory as possible
     * available again. Normally this method calls the various <code>release()</code> methods
     * of the factory class and is used always if the application reaches a certain amount
     * of consumed memory. In critical situations the application can use this method in
     * order to keep it running or at least store the current configurations or data portions
     * safely (panic mode).
     *
     * @since 1.6
     */
    //@Override
    public synchronized void releaseForced() {
/*
        // Release only free clients in the pool!
        Enumeration<String> en = this.cachePool.keys();  // NeverStrategy is not used here
        while (en.hasMoreElements()) {
            String url = en.nextElement();
            this.release(url);
        }
 */
    }

    /**
     * Release managed shared objects according to the configured strategy (free clients here).
     * <p>
     * For applications it may be useful to flush its caches but keep certain objects/resources
     * untouched since they will be needed in the future or the recreation would be expensive
     * (e.g. when contents must be loaded from a database or from the internet).
     * <p>
     * This method should be used by a periodically executed release task in order to release
     * all shared objects with timeout strategy.
     *
     * @since 1.6
     */
    //@Override
    public synchronized void releaseByStrategy() {
/*
        // Release only free clients in the pool where the timeout has been reached
        Enumeration<String> en = this.cachePool.keys();
        while (en.hasMoreElements()) {
            String url = en.nextElement();
            Vector<ManagedObject> pool = this.cachePool.get(url);

            Vector<ManagedObject> toremove = new Vector<ManagedObject>(0);
            for (ManagedObject managed : pool) {
                ClientWrapper client = (ClientWrapper) managed.object;

                // Let the release strategy object do its check here
                if (client != null && !this.cacheLocked.contains(client) &&   // Must not be locked!
                        managed.strategy != null && managed.strategy.canRelease(managed)) {  // Timeout reached

                    // The object can be released!
                    toremove.addElement(managed);
                }
            }
            for (ManagedObject managed : toremove) {  // Avoid exceptions

                // Disconnect from server first
                ClientWrapper client = (ClientWrapper) managed.object;
                client.disconnectClient();
                pool.remove(managed);
                VFSLibSettings.log(Level.CONFIG, "Released object " + client + " by strategy");
            }
            toremove.removeAllElements();
        }
 */
    }

    /**
     * Perform optional cleanup tasks like closing network connections or disposing dialogs.
     *
     * @param id The factory-unique identifier of the shared object
     * @since 1.6
     */
    //@Override
    public synchronized void cleanup(Object id) {
        // Nothing to do here
    }

    /**
     * Create the factory shared object, recycle if already existing.
     *
     * @param id The factory-unique identifier of the shared object
     * @return Does a shared instance already exist?
     * @since 1.6
     */
    //@Override
    public synchronized boolean hasSharedInstance(Object id) {

        String url = id instanceof String ? (String) id : String.valueOf(id);
/*
        // There must be an available client (not locked)
        Vector<ManagedObject> pool = this.cachePool.get(url);
        if (pool != null && !pool.isEmpty()) {
            for (ManagedObject managed : pool) {
                ClientWrapper client = (ClientWrapper) managed.object;
                if (!this.cacheLocked.contains(client)) return true;
            }
        }

 */
        return false;
    }

    /**
     * Provides the currently managed objects.
     *
     * @since 1.6
     */
    //@Override
    public synchronized /*ManagedObject[]*/void getRegisteredObjects() {
/*
        Vector<ManagedObject> sorter = new Vector<ManagedObject>(0);

        // The managed objects are sorted alphabetically by identifier
        Enumeration<String> en = this.cachePool.keys();
        while (en.hasMoreElements()) {
            String url = en.nextElement();
            Vector<ManagedObject> pool = this.cachePool.get(url);
            for (ManagedObject managed : pool) {  // Also locked instances
                sorter.addElement(managed);
            }
        }
        Comparator<ManagedObject> comparator = (o1, o2) -> {

            String name1 = ((ClientWrapper) o1.object).getURLID();
            String name2 = ((ClientWrapper) o2.object).getURLID();
            return name1.compareTo(name2);
        };
        Collections.sort(sorter, comparator);

        int size = sorter.size();
        ManagedObject[] array = new ManagedObject[size];
        for (int i = 0; i < size; i++) array[i] = sorter.elementAt(i);
        sorter.removeAllElements();

        return array;

 */
    }

    /**
     * Sets an object for an existing managed object.
     *
     * @param id  The factory-unique identifier of the shared object
     * @param obj The new object
     * @return The overwritten object, may be <code>null</code>
     * @throws NullPointerException If a parameter is <code>null</code>
     * @since 1.6
     */
    public synchronized Object set(Object id, Object obj) {
        return null;  // Not possible here
    }

    /**
     * Convenience method to show the current client pool.
     *
     * @return The text
     * @since 1.6
     */
    public String debugPrint() {
/*
        Object[][] data = new Object[0][6];
        OrderedHashtable<String, StackTraceElement[]> cacheStacks = new OrderedHashtable<String, StackTraceElement[]>(0);
        StringBuilder builder = new StringBuilder(0);

        Enumeration<String> en = this.cachePool.keys();
        while (en.hasMoreElements()) {
            String url = en.nextElement();
            Vector<ManagedObject> pool = this.cachePool.get(url);

            for (int i = 0; i < pool.size(); i++) {
                ManagedObject managed = pool.elementAt(i);
                ClientWrapper client = (ClientWrapper) managed.object;
                boolean locked = this.cacheLocked.contains(client);
                String hashcode = Integer.toHexString(client.hashCode());
                if (locked) cacheStacks.put(hashcode, this.cacheLockedBy.get(client));

                Object[] row = new Object[]{
                        client.getURLID(),                                             // ftp://anonymous@company.com:21
                        client.getClass().getName(),                                   // FTPClientWrapper
                        hashcode,                                                      // 1de0c09
                        String.valueOf(i + 1) + '/' + pool.size(),                     // 1/1
                        locked ? "locked" : "free"                                     // locked/free
                };
                data = (Object[][]) JavaUtils.addToArray(data, row);
            }
        }
        if (data.length >= 1) {
            builder.append("\n" + StringUtils.assembleTable(data, 5, null));

            en = cacheStacks.keys();
            while (en.hasMoreElements()) {
                String hashcode = en.nextElement();
                StackTraceElement[] trace = cacheStacks.get(hashcode);
                builder.append("\nStack trace for client " + hashcode + ":\n");

                // See Throwable.printStackTrace()
                for (int i = 0; i < trace.length; i++) builder.append("\tat " + trace[i] + '\n');
            }
        } else builder.append("\nNo open VFS clients available\n");
        cacheStacks.clear();

        return builder.toString();

 */
        return "";
    }
}
