/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.lang;

import java.lang.reflect.Array;


/**
 * This class holds various methods for common use.
 * <p>
 * All members of this class are declared public, most are static, too.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class JavaUtils {


    /**
     * Provides the linefeed appropriate for the system.
     */
    public final static String LINE_SEPARATOR = System.getProperty("line.separator", "\n");

    /**
     * Constructor method for i18n purposes only.
     *
     * @throws InstantiationException Error indication
     * @since 1.6
     */
    public JavaUtils() throws InstantiationException {

        if (!java.beans.Beans.isDesignTime()) {
            throw new InstantiationException("Do not use this constructor!");
        }
    }

    /**
     * Adds another element to the given array (saves <code>System.arraycopy()</code> calls).
     *
     * @param array  Target array of arbitrary type
     * @param newval Element to add, may be <code>null</code>
     * @return Expanded array
     * @throws IllegalArgumentException Parameters are invalid
     * @since 1.6
     */
    public static Object addToArray(Object array, Object newval) {

        if (!array.getClass().isArray()) {
            throw new IllegalArgumentException("First parameter must be an array");
        }

        // Normalize to primitive type if necessary
        Class arrayclazz = array.getClass().getComponentType();
        if (!arrayclazz.isPrimitive()) arrayclazz = JavaUtils.mapToPrimitive(arrayclazz);

        if (newval != null) {
            Class newvalclazz = newval.getClass();
            if (!newvalclazz.isPrimitive()) newvalclazz = JavaUtils.mapToPrimitive(newvalclazz);

            if (!arrayclazz.isAssignableFrom(newvalclazz)) {
                String msg = "Incompatible data types: " + arrayclazz.getName() + " != " + newvalclazz.getName();
                throw new IllegalArgumentException(msg);
            }
        }

        // New array needs to be created via reflection
        arrayclazz = array.getClass().getComponentType();
        Object temp = Array.newInstance(arrayclazz, Array.getLength(array) + 1);
        System.arraycopy(array, 0, temp, 0, Array.getLength(array));
        Array.set(temp, Array.getLength(array), newval);
        return temp;
    }

    /**
     * Checks if the given object is element of the given array (equals).
     *
     * @param object The object to be checked
     * @param array  The array to check against
     * @return Is the object element of the array?
     * @throws NullPointerException If object and/or array is <code>null</code>
     * @since 1.6
     */
    public static boolean elementOf(Object object, Object array) {

        Object arrayobj;

        if (array.getClass().isArray()) {

            // For the primitive types we use auto-boxing, so let's compare objects
            for (int i = 0; i < Array.getLength(array); i++) {
                arrayobj = Array.get(array, i);
                if (arrayobj == null) continue;

                // Changed to compare File and Win32Folder objects (subclass of File)
                if (arrayobj.equals(object)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Provides the associated primitive type (<code>Integer</code> ... <code>int</code>).
     * <p>
     * The types boolean, byte, char, short, int, long, float, double, and void are supported.
     *
     * @param clazz Class to be mapped
     * @return Primitive type or value of <code>clazz</code> if not available
     * @since 1.6
     */
    public static Class mapToPrimitive(Class clazz) {

        if (clazz.equals(Boolean.class)) return Boolean.TYPE;
        else if (clazz.equals(Byte.class)) return Byte.TYPE;
        else if (clazz.equals(Character.class)) return Character.TYPE;
        else if (clazz.equals(Short.class)) return Short.TYPE;
        else if (clazz.equals(Integer.class)) return Integer.TYPE;
        else if (clazz.equals(Long.class)) return Long.TYPE;
        else if (clazz.equals(Float.class)) return Float.TYPE;
        else if (clazz.equals(Double.class)) return Double.TYPE;
        else if (clazz.equals(Void.class)) return Void.TYPE;
        return clazz;
    }
}
