/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.gdrive;

import org.apache.commons.vfs2.FileSystem;
import org.apache.commons.vfs2.FileSystemConfigBuilder;
import org.apache.commons.vfs2.FileSystemOptions;


/**
 * The config builder for the various Google Drive configuration options.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class GDriveFileSystemConfigBuilder extends FileSystemConfigBuilder {


    /**
     * Shared <code>GDriveFileSystemConfigBuilder</code> instance.
     */
    protected static GDriveFileSystemConfigBuilder sharedInstance;

    /**
     * Variable name.
     */
    public final static String DISPLAY_NAME = GDriveFileSystemConfigBuilder.class.getName() + ".DISPLAY_NAME";
    /**
     * Variable name.
     */
    public final static String ACCESS_TOKEN = GDriveFileSystemConfigBuilder.class.getName() + ".ACCESS_TOKEN";
    /**
     * Variable name.
     */
    public final static String USE_TRASH = GDriveFileSystemConfigBuilder.class.getName() + ".USE_TRASH";
    /**
     * Variable name.
     */
    public final static String CONNECT_TIMEOUT_SECONDS = GDriveFileSystemConfigBuilder.class.getName() + ".CONNECT_TIMEOUT_SECONDS";
    /**
     * Variable name.
     */
    public final static String READ_TIMEOUT_SECONDS = GDriveFileSystemConfigBuilder.class.getName() + ".READ_TIMEOUT_SECONDS";

    /**
     * Constructor method.
     *
     * @since 1.6
     */
    public GDriveFileSystemConfigBuilder() {
        super("gdrive.");  // Static to find resources, do not use dynamic scheme here
    }

    /**
     * Provides the shared instance of the factory implementation.
     *
     * @return Shared instance
     * @since 1.6
     */
    public static synchronized GDriveFileSystemConfigBuilder getSharedInstance() {

        if (sharedInstance == null) sharedInstance = new GDriveFileSystemConfigBuilder();
        return sharedInstance;
    }

    /**
     * Provides the file system class to be configured by this builder.
     *
     * @return File system class
     * @since 1.6
     */
    @Override
    protected Class<? extends FileSystem> getConfigClass() {
        return GDriveFileSystem.class;
    }

    /**
     * Setter for Google Drive account display name.
     *
     * @param options The options
     * @param name    The display name
     * @since 1.6
     */
    public void setAccountDisplayName(FileSystemOptions options, String name) {
        super.setParam(options, DISPLAY_NAME, name);
    }

    /**
     * Getter for Google Drive account display name.
     *
     * @param options The options
     * @return The display name, may be <code>null</code>
     * @since 1.6
     */
    public String getAccountDisplayName(FileSystemOptions options) {
        return super.getParam(options, DISPLAY_NAME);
    }

    /**
     * Setter for Google Drive access token.
     *
     * @param options The options
     * @param token   The access token
     * @since 1.6
     */
    public void setAccessToken(FileSystemOptions options, String token) {
        super.setParam(options, ACCESS_TOKEN, token);
    }

    /**
     * Getter for Google Drive access token.
     *
     * @param options The options
     * @return The access token, may be <code>null</code>
     * @since 1.6
     */
    public String getAccessToken(FileSystemOptions options) {
        return super.getParam(options, ACCESS_TOKEN);
    }

    /**
     * Setter for delete mode: put into trash (may be restored) or delete permanently.
     *
     * @param options The options
     * @param trash   Move to trash?
     * @since 1.6
     */
    public void setUseTrash(FileSystemOptions options, Boolean trash) {
        super.setParam(options, USE_TRASH, trash);
    }

    /**
     * Getter for delete mode: put into trash (may be restored) or delete permanently.
     *
     * @param options The options
     * @return Put into trash? May be <code>null</code>.
     * @since 1.6
     */
    public Boolean getUseTrash(FileSystemOptions options) {
        return super.getParam(options, USE_TRASH);
    }

    /**
     * Setter for connect timeout in seconds.
     *
     * @param options The options
     * @param timeout Seconds
     * @since 2.8
     */
    public void setConnectTimeoutSeconds(FileSystemOptions options, Integer timeout) {
        super.setParam(options, CONNECT_TIMEOUT_SECONDS, timeout);
    }

    /**
     * Getter for connect timeout in seconds.
     *
     * @param options The options
     * @return Seconds, may be <code>null</code>.
     * @since 2.8
     */
    public Integer getConnectTimeoutSeconds(FileSystemOptions options) {
        return super.getParam(options, CONNECT_TIMEOUT_SECONDS);
    }

    /**
     * Setter for read timeout in seconds.
     *
     * @param options The options
     * @param timeout Seconds
     * @since 2.8
     */
    public void setReadTimeoutSeconds(FileSystemOptions options, Integer timeout) {
        super.setParam(options, READ_TIMEOUT_SECONDS, timeout);
    }

    /**
     * Getter for read timeout in seconds.
     *
     * @param options The options
     * @return Seconds, may be <code>null</code>.
     * @since 2.8
     */
    public Integer getReadTimeoutSeconds(FileSystemOptions options) {
        return super.getParam(options, READ_TIMEOUT_SECONDS);
    }
}
