/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.gdrive;

import com.lf.vfslib.VFSLib;
import org.apache.commons.vfs2.*;
import org.apache.commons.vfs2.provider.AbstractFileName;
import org.apache.commons.vfs2.provider.AbstractFileSystem;
import org.apache.commons.vfs2.provider.GenericFileName;

import java.io.IOException;
import java.util.Collection;


/**
 * Represents the files hosted by Google Drive.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class GDriveFileSystem extends AbstractFileSystem implements FileSystem {


    /**
     * The parental object.
     */
    protected VFSLib vfsLib;
    /**
     * The file provider.
     */
    protected GDriveFileProvider provider;


    /**
     * The constructor method for this file system.
     *
     * @param rootName The root name
     * @param options  The file system options
     * @param vfslib   The parental object, shared instance if <code>null</code>
     * @param provider The file provider
     * @since 1.6
     */
    protected GDriveFileSystem(GenericFileName rootName, FileSystemOptions options, VFSLib vfslib, GDriveFileProvider provider) {

        super(rootName, null, options);

        // Using this info an authorize URL can be acquired from Google
        // in order to let the user allow this application to access his Google Drive resources.
        this.vfsLib = vfslib != null ? vfslib : VFSLib.getSharedInstance();
        this.provider = provider;
    }

    /**
     * Returns a Google Drive client to access the server.
     *
     * @return The client
     * @throws java.io.IOException If an I/O error occurs
     * @since 1.6
     */
    protected GDriveClientWrapper getClient() throws IOException {

        // Get client from pool, create fresh one otherwise
        return GDriveClientWrapper.getGDriveClientWrapper((GenericFileName) getRoot().getName(), getFileSystemOptions(),
                this.vfsLib, this.provider);
    }

    /**
     * Returns a Google Drive client to the pool.
     *
     * @param client The client
     * @since 1.6
     */
    protected void putClient(GDriveClientWrapper client) {

        // Release client from pool
        GDriveClientWrapper.unlockGDriveClientWrapper(client);
    }

    /**
     * Adds the capabilities of this file system.
     *
     * @param caps The capabilities
     * @since 1.6
     */
    @Override
    protected void addCapabilities(Collection<Capability> caps) {
        caps.addAll(GDriveFileProvider.capabilities);
    }

    /**
     * Creates a file object.
     * <p>
     * This method is called only if the requested. File is not cached.
     *
     * @param name The capabilities
     * @return The file object
     * @since 1.6
     */
    @Override
    protected FileObject createFile(AbstractFileName name) throws FileSystemException {
        return new GDriveFileObject(name, this);
    }

    /**
     * Last mod time is only a int and in seconds, thus can be off by 999.
     *
     * @return 1000
     * @since 1.6
     */
    @Override
    public double getLastModTimeAccuracy() {
        return 1000L;
    }

    /**
     * Provides the VFSLib instance for this filesystem.
     *
     * @return The instance
     * @since 1.6
     */
    public VFSLib getVFSLib() {
        return this.vfsLib;
    }

    /**
     * Provides the provider for this filesystem.
     *
     * @return The provider
     * @since 1.6
     */
    public GDriveFileProvider getProvider() {
        return this.provider;
    }
}
