/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.gdrive;

import com.google.api.client.googleapis.auth.oauth2.GoogleAuthorizationCodeFlow;
import com.lf.vfslib.VFSLib;
import org.apache.commons.vfs2.*;
import org.apache.commons.vfs2.provider.AbstractOriginatingFileProvider;
import org.apache.commons.vfs2.provider.GenericFileName;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;


/**
 * A provider to access the files hosted by Google Drive.
 * <p>
 * This class coordinates the whole Google Drive functionality used by VFS2.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class GDriveFileProvider extends AbstractOriginatingFileProvider {


    /**
     * The Google authorization flow.
     */
    protected final GoogleAuthorizationCodeFlow authFlow;
    /**
     * The application name.
     */
    protected final String appName;
    /**
     * The parental object.
     */
    protected final VFSLib vfsLib;


    /**
     * The provider's capabilities.
     */
    static final Collection<Capability> capabilities =
            Collections.unmodifiableCollection(Arrays.asList(
                    //Capability.APPEND_CONTENT,
                    //Capability.ATTRIBUTES,
                    //Capability.COMPRESS,
                    Capability.CREATE,
                    Capability.DELETE,
                    //Capability.DIRECTORY_READ_CONTENT,
                    //Capability.DISPATCHER,
                    //Capability.FS_ATTRIBUTES,
                    Capability.GET_LAST_MODIFIED, // Currently not supported for folders
                    Capability.GET_TYPE,
                    //Capability.JUNCTIONS,
                    //Capability.LAST_MODIFIED,
                    Capability.LIST_CHILDREN,
                    //Capability.MANIFEST_ATTRIBUTES,
                    Capability.RANDOM_ACCESS_READ,
                    //Capability.RANDOM_ACCESS_WRITE,
                    Capability.READ_CONTENT,
                    Capability.RENAME,
                    Capability.SET_LAST_MODIFIED_FILE,
                    Capability.SET_LAST_MODIFIED_FOLDER,
                    //Capability.SIGNING,
                    Capability.URI,
                    //Capability.VIRTUAL,
                    Capability.WRITE_CONTENT));


    /**
     * Constructor method (do not use).
     *
     * @throws InstantiationException Error indication
     * @since 1.6
     */
    public GDriveFileProvider() throws InstantiationException {
        throw new InstantiationException("Do not use this constructor!");
    }

    /**
     * Constructor method.
     *
     * @param authflow The Google authorization flow (required, type is <code>com.google.api.client.googleapis.auth.oauth2.GoogleAuthorizationCodeFlow</code>)
     * @param appname  The application name (required)
     * @param vfslib   The parental object
     * @throws NullPointerException     If parameters are <code>null</code>
     * @throws IllegalArgumentException If a parameter is not of the expected type
     * @since 1.6
     */
    public GDriveFileProvider(Object authflow, String appname, VFSLib vfslib) {

        super();

        if (authflow == null) throw new NullPointerException();
        if (appname == null) throw new NullPointerException();
        if (!(authflow instanceof GoogleAuthorizationCodeFlow)) {
            throw new IllegalArgumentException("Parameter authflow must be of type GoogleAuthorizationCodeFlow");
        }
        if (vfslib == null) throw new NullPointerException();

        // Using this info an authorize URL can be acquired from Google
        // in order to let the user allow this application to access his Google Drive resources.
        this.authFlow = (GoogleAuthorizationCodeFlow) authflow;
        this.appName = appname;
        this.vfsLib = vfslib;

        setFileNameParser(GDriveFileNameParser.getSharedInstance());
    }

    /**
     * Provides the provider's Google Drive authorization flow.
     *
     * @return The auth flow
     * @since 1.6
     */
    public GoogleAuthorizationCodeFlow getAuthFlow() {
        return this.authFlow;
    }

    /**
     * Return the name of the application.
     *
     * @return The name
     * @since 1.6
     */
    public String getApplicationName() {
        return this.appName;
    }

    /**
     * Creates a Google Drive file system.
     *
     * @param name    The file name
     * @param options The file system options
     * @throws org.apache.commons.vfs2.FileSystemException If something goes wrong
     * @since 1.6
     */
    @Override
    protected FileSystem doCreateFileSystem(FileName name, FileSystemOptions options) throws FileSystemException {

        // Create the file system
        GenericFileName rootName = (GenericFileName) name;

        // Check out for a free connection here to detect errors
        GDriveClientWrapper client = GDriveClientWrapper.getGDriveClientWrapper(rootName, options, this.vfsLib, this);
        GDriveClientWrapper.unlockGDriveClientWrapper(client);

        return new GDriveFileSystem(rootName, options, this.vfsLib, this);
    }

    /**
     * Provides the builder instance responsible for proper file system options.
     *
     * @return The builder
     * @since 1.6
     */
    @Override
    public FileSystemConfigBuilder getConfigBuilder() {
        return GDriveFileSystemConfigBuilder.getSharedInstance();
    }

    /**
     * Provides the provider's capabilities.
     *
     * @return The collection
     * @since 1.6
     */
    public Collection<Capability> getCapabilities() {
        return GDriveFileProvider.capabilities;
    }
}
