/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.gdrive;

import com.google.api.client.http.InputStreamContent;
import com.google.api.client.util.DateTime;
import com.google.api.services.drive.Drive;
import com.google.api.services.drive.model.File;
import com.google.api.services.drive.model.FileList;
import com.lf.vfslib.VFSLib;
import com.lf.vfslib.core.VFSLibConstants;
import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.dropbox.DbxClientWrapper;
import com.lf.vfslib.io.VFSRandomAccessContent;
import com.lf.vfslib.lang.JavaUtils;
import org.apache.commons.vfs2.*;
import org.apache.commons.vfs2.provider.AbstractFileName;
import org.apache.commons.vfs2.provider.AbstractFileObject;
import org.apache.commons.vfs2.provider.UriParser;
import org.apache.commons.vfs2.util.MonitorInputStream;
import org.apache.commons.vfs2.util.MonitorOutputStream;
import org.apache.commons.vfs2.util.RandomAccessMode;

import java.io.*;
import java.lang.reflect.Field;
import java.util.*;
import java.util.logging.Level;


/**
 * Represents a Google Drive file or folder.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class GDriveFileObject extends AbstractFileObject {

    protected final static String ROOT_ALIAS = "root";
    protected final static String FOLDER_MIME_TYPE = "application/vnd.google-apps.folder";
    protected final static String RELEVANT_FIELDS = "files(id,name,mimeType,modifiedTime,size,parents,trashed)";
    protected final static int PAGE_SIZE = 100;


    /**
     * The underlying file system.
     */
    final protected GDriveFileSystem fileSystem;
    /**
     * The relative path on the Google Drive server.
     */
    protected String relPath;
    /**
     * Helper to avoid multiple refreshing.
     */
    protected boolean inRefresh = false;
    /**
     * The underlying Google Drive file/folder.
     */
    protected File googleFile = null;
    /**
     * The shared ID of the Google Drive root folder.
     */
    protected String rootID = null;
    /**
     * The attributes specific for cloud file systems (e.g. "Content-Length" for proper upload).
     */
    protected Map<String, Object> attributes = null;


    /**
     * Constructor method.
     *
     * @param name       The file name
     * @param filesystem The file system
     * @throws NullPointerException                        If a parameter is <code>null</code>
     * @throws org.apache.commons.vfs2.FileSystemException If something goes wrong
     * @since 1.6
     */
    public GDriveFileObject(AbstractFileName name, GDriveFileSystem filesystem) throws FileSystemException {

        super(name, filesystem);

        // VFS2 uses URI's internally. Possible name normalizations for Google Drive are:

        /*
           Type                 Value of name                                                 Relative name               relPath (normalized)        googleFile
           ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------
           Root folder          gdrive://johndoe@drive.google.com/                            "." or "/"                  "/"                         name=Meine Ablage (for Germany)
           File in root         gdrive://johndoe@drive.google.com/battery_96.png              battery_96.png              battery_96.png              name=battery_96.png
           Subfolder            gdrive://johndoe@drive.google.com/testfolder/                 testfolder/                 /testfolder                 name=testfolder
           File in subfolder    gdrive://johndoe@drive.google.com/testfolder/check2_16.png    testfolder/check2_16.png    /testfolder/check2_16.png   name=check2_16.png
         */

        // See statSelf() method on how the googleFile is resolved via Google Drive API.

        // Remember that the "gdrive:" protocol is virtual only and cannot be queried e.g. in a browser.
        // The trash mode etc. are embedded in the connection options and not part of the VFS2 URI.

        this.fileSystem = filesystem;
        this.relPath = UriParser.decode(filesystem.getRootName().getRelativeName(name));
        if (this.relPath.equals(".")) this.relPath = "/";  // Does not work with Google Drive!
        else if (!this.relPath.startsWith("/")) this.relPath = '/' + this.relPath;  // Files
    }

    /**
     * Clean-up method to help the gc.
     *
     * @throws Throwable Error indication
     * @since 1.6
     */
    @Override
    protected void finalize() throws Throwable {

        super.finalize();

        if (this.attributes != null) this.attributes.clear();
    }

    /**
     * Checks if the current relative path represents the root folder.
     *
     * @return Root folder?
     * @since 2.8
     */
    protected boolean isRoot() {
        return this.relPath.equals("/");
    }

    /**
     * Translate the absolute path for URL to a relative path compatible with Google Drive.
     *
     * @return The Google path
     * @since 1.6
     */
    protected String getGooglePath() {

        if (isRoot()) return "";
        return this.relPath.substring(1);  // "/Marianne Hoppe" -> "Marianne Hoppe"
    }

    /**
     * Translate the absolute path for URL to a relative path compatible with Google Drive.
     *
     * @return The Google name
     * @since 1.6
     */
    protected String getGoogleName() {

        // If relPath is "/Marianne Hoppe/Bild.jpeg" , the title is simply "Bild.jpeg"
        if (isRoot()) return ROOT_ALIAS;
        return this.relPath.substring(this.relPath.lastIndexOf('/') + 1);
    }

    /**
     * Checks if the current relative path denotes a file in the Google Drive root folder.
     *
     * @return Root subentry?
     * @since 1.6
     */
    protected boolean isInRoot() {

        if (this.isRoot()) return true;
        return this.relPath.lastIndexOf('/') == 0;
        // "/Marianne Hoppe" is
        // "/Marianne Hoppe/Bild.jpeg" is not
    }

    /**
     * Provides the ID of the Google Drive root folder (stored separately, recycled).
     *
     * @return Root ID, <code>null</code> if not available
     * @since 1.6
     */
    protected String getRootID() {

        if (this.rootID == null) {  // Read only once (static for one account)
            try {
                GDriveClientWrapper client = this.fileSystem.getClient();

                // Get root folder ID (stored separately)
                this.rootID = client.driveClient.files().get(ROOT_ALIAS).setFields("id").execute().getId();
                // 0APouqTse0V0nUk9PVA
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            }
        }
        return this.rootID;
    }

    /**
     * Detach from the file system.
     *
     * @since 1.6
     */
    @Override
    protected void doDetach() throws Exception {
        this.googleFile = null;
    }

    /**
     * Refresh this file.
     *
     * @throws org.apache.commons.vfs2.FileSystemException If something goes wrong
     * @since 1.6
     */
    @Override
    public void refresh() throws FileSystemException {

        if (!this.inRefresh) {  // Avoid redundant refreshing
            try {
                this.inRefresh = true;
                super.refresh();
            } finally {
                this.inRefresh = false;
            }
        }
    }

    /**
     * Determines the type of this file, returns <code>null</code> if the file does not exist.
     *
     * @return The file type
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected FileType doGetType() throws Exception {

        statSelf();

        // May still be null if an anticipated file is queried (e.g. abc.txt.sha)
        if (this.googleFile != null) {  // Has been resolved (exists)?

            // Special MIME type for folders
            if (this.googleFile.getMimeType().equals(FOLDER_MIME_TYPE)) return FileType.FOLDER;
            else return FileType.FILE;
        }
        FileType type = FileType.IMAGINARY; // File/folder does not exist
        try {
            // This is special here: IMAGINARY has no attributes, allow attributes for setting upload "Content-Length"
            Field field = type.getClass().getDeclaredField("hasAttrs");  // enum
            field.setAccessible(true);  // Override access limitations for private, protected etc.
            field.set(type, true);
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        return type;
    }

    /**
     * Determines the type of this file.
     *
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    protected void statSelf() throws Exception {

        if (this.googleFile != null) return;  // Already resolved

        GDriveClientWrapper client = this.fileSystem.getClient();
        try {
            // Special MIME type for folders is "application/vnd.google-apps.folder"
            if (this.isRoot()) {
                Drive.Files.Get request = client.driveClient.files().get(this.getRootID());
                this.googleFile = request.execute();
            } else {
                // Get the IDs of the parent folders for subentry.
                // Google Drive does not allow to query full path like "Marianne Hoppe/Postkarte.jpeg" at once.
                String parentid = this.getRootID();  // Start from Google Drive root

                // Marianne Hoppe
                // Marianne Hoppe/subfolder
                // Marianne Hoppe/subfolder/Postkarte.jpeg

                String[] paths = this.getGooglePath().split("([/])");
                for (int i = 0; i < paths.length; i++) {
                    String next = paths[i];

                    // Get ID of next subfolder
                    String q = "name='" + next + "' and trashed=false and '" + parentid + "' in parents";
                    List<File> list = listAllFiles(q);
                    if (list.size() == 1) {
                        if (i == (paths.length - 1)) this.googleFile = list.get(0);  // Last index = desired file
                        else parentid = list.get(0).getId();
                    } else break;  // Parent does not exists
                }
            }
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e); // Needed by developers
            this.googleFile = null;  // Does not exist
        } finally {
            this.fileSystem.putClient(client);
        }
    }

    /**
     * Called when the type or content of this file changes.
     *
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected void onChange() throws Exception {

        this.googleFile = null;
        statSelf();
    }

    /**
     * Creates this file as a folder.
     *
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected void doCreateFolder() throws Exception {

        this.statSelf();

        if (!isRoot()) {
            GDriveClientWrapper client = this.fileSystem.getClient();
            try {
                // Caution: Google Drive allows creating multiple folders with the same name!
                if (doGetType().equals(FileType.IMAGINARY)) {  // Create only if folder does not exist
                    // Caution: Google Drive allows creating a folder like "Marianne Hoppe/Unterordner"!
                    this.doCreateEntry(client, FOLDER_MIME_TYPE);
                    return;
                }
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                this.fileSystem.putClient(client);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_CREATE_FOLDER_ERROR"));  // Recycled
    }

    /**
     * Gets the last modification date and time.
     *
     * @return The last modification time in milliseconds
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected long doGetLastModifiedTime() throws Exception {

        this.statSelf();

        // May be null if an anticipated file is queried (e.g. abc.txt.sha)
        try {
            DateTime modtime = this.googleFile.getModifiedTime();
            if (modtime != null) return modtime.getValue();
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_UNKNOWN_MOD_TIME_ERROR"));  // Recycled
    }

    /**
     * Sets the last modified time of this file.
     *
     * @param modtime Last modification time in milliseconds
     * @return Successful?
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected boolean doSetLastModifiedTime(long modtime) throws Exception {

        this.statSelf();

        GDriveClientWrapper client = this.fileSystem.getClient();

        // May be null if an anticipated file is queried (e.g. abc.txt.sha)
        try {
            File file = new File();
            file.setModifiedTime(new DateTime(modtime));
            client.driveClient.files().update(this.googleFile.getId(), file).execute();
            onChange();
            return true;
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        } finally {
            this.fileSystem.putClient(client);
        }
        throw new FileSystemException(VFSLibSettings.getUserText(GDriveClientWrapper.class.getName() + "_SET_MOD_TIME_ERROR"));
    }

    /**
     * Deletes the file.
     * <p>
     * Entries either can be put into trash (may be restored) or be deleted permanently here.
     *
     * @throws Exception If something goes wrong
     * @see GDriveFileSystemConfigBuilder#setUseTrash(FileSystemOptions, Boolean)
     * @since 1.6
     */
    @Override
    protected void doDelete() throws Exception {

        this.statSelf();

        if (!isRoot()) {
            GDriveClientWrapper client = this.fileSystem.getClient();
            try {
                FileSystemOptions fsoptions = this.fileSystem.getFileSystemOptions();
                Boolean usetrash = GDriveFileSystemConfigBuilder.getSharedInstance().getUseTrash(fsoptions);

                if (usetrash == null || usetrash) {  // Default: use trash
                    // Allow entry be restored
                    // Found here: https://stackoverflow.com/questions/40286246/google-drive-rest-api-v3-how-to-move-a-file-to-the-trash
                    File file = new File();
                    file.setTrashed(true);
                    client.driveClient.files().update(this.googleFile.getId(), file).execute();
                } else {  // Delete permanently
                    Drive.Files.Delete request = client.driveClient.files().delete(this.googleFile.getId());
                    request.execute();
                }
                return;
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                this.fileSystem.putClient(client);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(GDriveClientWrapper.class.getName() + "_DELETE_ERROR"));
    }

    /**
     * Rename the file.
     * <p>
     * This may take a while that Google Drive has finished renaming the file/folder internally.
     * So please give it a little timeout before <code>exists()</code> etc. is called.
     *
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected void doRename(FileObject newfile) throws Exception {

        this.statSelf();

        if (!isRoot()) {
            GDriveClientWrapper client = this.fileSystem.getClient();
            try {
                // Found here: https://developers.google.com/drive/v2/reference/files/patch
                File file = new File();
                file.setName(newfile.getName().getBaseName());
                client.driveClient.files().update(this.googleFile.getId(), file).execute();
                return;
                // May take a while that exists() returns true

            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                this.fileSystem.putClient(client);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_RENAME_ERROR"));  // Recycled
    }

    /**
     * Lists the children of this file.
     *
     * @return The list of children, may be empty
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected FileObject[] doListChildrenResolved() throws Exception {

        FileObject[] array = FileObject.EMPTY_ARRAY;
        GDriveClientWrapper client = this.fileSystem.getClient();
        FileSystemManager fsmanager = getFileSystem().getFileSystemManager();

        statSelf();

        try {
            String folderID = (this.isRoot() ? this.getRootID() : this.googleFile.getId());
            String q = "trashed=false and parents in '" + folderID + "'";
            List<File> list = listAllFiles(q);
            // Listed like by https://drive.google.com/drive/my-drive

            // Convert to array with VFS files
            for (File child : list) {
                FileObject fileobj = this.fileSystem.resolveFile( // E.g. "/Erste Schritte.pdf"
                        fsmanager.resolveName(getName(), UriParser.encode(child.getName()), NameScope.CHILD));
                array = (FileObject[]) JavaUtils.addToArray(array, fileobj);

                // Populate required fields, no need to call statSelf() again
                ((GDriveFileObject) fileobj).googleFile = child;
            }
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
            throw e;
        } finally {
            this.fileSystem.putClient(client);
        }
        return array;
    }

    /**
     * Lists the children of this file.
     *
     * @return The list of children, may be empty
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected String[] doListChildren() throws Exception {

        FileObject[] children = this.doListChildrenResolved();
        return Arrays.stream(children).map(String::valueOf).toArray(String[]::new);
    }

    /**
     * Returns the size of the file content (in bytes).
     *
     * @return The size, -1 if not available
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected long doGetContentSize() throws Exception {

        statSelf();

        // May be null if an anticipated file is queried (e.g. abc.txt.sha)
        try {
            return this.googleFile.getSize();  // null if entry has no content (e.g. folder)
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_UNKNOWN_SIZE_ERROR"));  // Recycled
    }

    /**
     * Provides the instance to model random access for files.
     *
     * @return The instance
     * @since 1.6
     */
    @Override
    protected RandomAccessContent doGetRandomAccessContent(RandomAccessMode mode) {
        return new VFSRandomAccessContent(this, mode);
    }

    /**
     * Creates an input stream to read the file content from.
     *
     * @return The input stream
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected InputStream doGetInputStream() throws Exception {

        this.statSelf();

        if (!FileType.FILE.equals(getType())) { // Required to exist
            String text = VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_READ_NOT_FILE_ERROR");  // Recycled
            throw new FileSystemException(text, getName());
        }

        GDriveClientWrapper client = this.fileSystem.getClient();

        // VFS-113: Avoid NPE
        synchronized (this.fileSystem) {
            try {
                // See https://developers.google.com/drive/api/guides/manage-downloads?hl=de

                // Download BLOB contents
                InputStream inputstream = client.driveClient.files() //
                        .get(this.googleFile.getId()) //
                        .executeMediaAsInputStream();
                // Adjust connect/read timeouts if SocketTimeoutException's occur for large files (default 15s)

                return new GDriveInputStream(client, inputstream);
            } finally {
                // For client release and stream close see GDriveOutputStream
            }
        }
    }

    /**
     * Creates an output stream to write the file content to.
     *
     * @param append Append? Create freshly otherwise.
     * @return The output stream, maybe <code>null</code>
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected OutputStream doGetOutputStream(boolean append) throws Exception {

        this.statSelf();

        if (!FileType.FILE.equals(getType()) && !FileType.IMAGINARY.equals(getType())) {
            String text = VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_WRITE_NOT_FILE_ERROR");  // Recycled
            throw new FileSystemException(text, getName());
        }

        GDriveClientWrapper client = this.fileSystem.getClient();

        // VFS-113: Avoid NPE
        synchronized (this.fileSystem) {
            try {
                // See https://developers.google.com/drive/api/guides/manage-uploads?hl=de#java

                PipedOutputStream postream = new PipedOutputStream();
                PipedInputStream pistream = new PipedInputStream(postream, 4096);  // See FileSystemCopier class

                // Amazon S3 Javadoc:
                // "When uploading directly from an input stream, content length must be specified before data
                // can be uploaded to Amazon S3. If not provided, the library will have to buffer the contents
                // of the input stream in order to calculate it. Amazon S3 explicitly requires that the content
                // length be sent in the request headers before any of the data is sent."

                // Means: If not provided the whole data from the stream is cached to get the content length (JavaHeapSpace error)!

                // Content length has been set as attribute for this file?
                long len = -1;
                try {
                    // Throws FileSystemException
                    len = Long.parseLong(String.valueOf(super.getContent().getAttribute(VFSLibConstants.ATTR_CONTENT_LENGTH)));
                } catch (Exception e) {
                    VFSLibSettings.log(Level.WARNING, e);
                }

                File fileMetadata = new File();
                fileMetadata.setName(this.getGoogleName());

                // Need the File ID of the parent folder to create a subfolder or subentry
                List<String> parentReferences = getParentReferences();

                // Do not create in wrong folder
                if (parentReferences != null) {
                    fileMetadata.setParents(parentReferences);
                }

                // Send the request to the API
                InputStreamContent content = new InputStreamContent(fileMetadata.getMimeType(), pistream);
                if (len != -1) content.setLength(len);
                final Drive.Files.Create request = client.driveClient.files().create(fileMetadata, content);

                Thread thread = new Thread(() -> {
                    try {
                        request.execute();
                    } catch (Exception e) {
                        VFSLibSettings.log(Level.WARNING, e);
                    }
                }, VFSLibSettings.LOG_PREFIX + "Update Requester");
                thread.setPriority(Thread.MIN_PRIORITY);
                thread.start();

                return new GDriveOutputStream(client, postream, thread);  // Release client afterwards
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                // For client release and stream close see GDriveOutputStream
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_OUTPUTSTREAM_ERROR"));  // Recycled
    }

    /**
     * An <code>InputStream</code> that monitors for end-of-file.
     *
     * @since 1.6
     */
    protected class GDriveInputStream extends MonitorInputStream {


        protected final GDriveClientWrapper client;

        public GDriveInputStream(GDriveClientWrapper client, InputStream in) {

            super(in);
            this.client = client;
        }

        @Override
        protected void onClose() {

            // No need to close stream here
            fileSystem.putClient(this.client);
        }
    }

    /**
     * An <code>OutputStream</code> that wraps a Google Drive <code>OutputStream</code>.
     *
     * @since 1.6
     */
    protected class GDriveOutputStream extends MonitorOutputStream {


        protected final GDriveClientWrapper client;
        protected final Thread requester;

        public GDriveOutputStream(GDriveClientWrapper client, OutputStream out, Thread requester) {

            super(out);
            this.client = client;
            this.requester = requester;
        }

        @Override
        protected void onClose() {

            try {
                // Wait for request thread to be finished, size does not fit otherwise etc.
                this.requester.join();
            } catch (Exception ignored) {
            }

            // No need to close stream here
            fileSystem.putClient(this.client);
        }
    }

    /**
     * Creates this entry (file or folder).
     *
     * @param client   The client
     * @param mimetype The MIME, use "application/vnd.google-apps.folder" for folders
     * @return The new file, <code>null</code> otherwise
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    protected File doCreateEntry(GDriveClientWrapper client, String mimetype) throws Exception {

        this.statSelf();

        // Found here: https://developers.google.com/drive/api/guides/folder?hl=de

        File body = new File();
        body.setName(getGoogleName());
        if (mimetype != null) body.setMimeType(mimetype);

        // Need the File ID of the parent folder to create a subfolder or subentry
        List<String> parentReferences = getParentReferences();

        // Do not create in wrong folder
        if (parentReferences != null) {
            body.setParents(parentReferences);
            return client.driveClient.files().create(body).execute();
        }
        return null;
    }

    /**
     * Provides the parent references of the
     *
     * @return The parent IDs
     * @since 2.8
     */
    protected List<String> getParentReferences() throws Exception {

        List<String> parentReferences = null;
        if (this.isInRoot()) {
            parentReferences = new Vector<>(1);
            parentReferences.add(this.getRootID());
        } else {  // Get subfolder of Google Drive first

            GDriveFileObject parent = (GDriveFileObject) this.getParent();
            parent.statSelf();  // Read ID etc.

            if (parent.googleFile != null) {  // Exists
                parentReferences = new Vector<>(1);
                parentReferences.add(parent.googleFile.getId());
            }
        }
        return parentReferences;
    }

    /**
     * Provides the VFSLib instance for this filesystem.
     *
     * @return The instance
     * @since 1.6
     */
    public VFSLib getVFSLib() {
        return this.fileSystem.getVFSLib();
    }

    /**
     * Provides the <code>ClientWrapper</code> implementation for this file system.
     *
     * @return The client wrapper holding the low-level client
     * @throws IOException If an I/O error occurs
     * @since 2.8
     */
    public GDriveClientWrapper getGDriveClientWrapper() throws IOException {
        return this.fileSystem.getClient();
    }

    /**
     * Returns the attributes specific for cloud file systems (e.g. "Content-Length" for proper upload)
     *
     * @return The map
     * @since 1.6
     */
    @Override
    protected Map<String, Object> doGetAttributes() {

        synchronized (this) {
            if (this.attributes == null) {
                this.attributes = new HashMap<>(0);
            }
        }
        return this.attributes;
    }

    /**
     * Sets an attribute of this cloud file.
     *
     * @param attrName The name (ATTR_CONTENT_LENGTH|...)
     * @param value    The value
     * @since 1.6
     */
    @Override
    protected void doSetAttribute(String attrName, Object value) throws Exception {

        doGetAttributes();  // Make sure cache exists
        this.attributes.put(attrName, value);
    }

    /**
     * Convenience method to list all Google Drive files at once from the available pages.
     *
     * @param omnipotentMemberOfQContinuum The query
     * @return The file list
     * @throws IOException If an I/O error occurs
     * @since 2.8
     */
    protected List<File> listAllFiles(String omnipotentMemberOfQContinuum) throws IOException {

        List<File> list = new ArrayList<>();
        String pageToken = null;
        GDriveClientWrapper client = this.fileSystem.getClient();

        do {
            Drive.Files.List request = client.driveClient.files().list();
            request.setPageSize(PAGE_SIZE);
            request.setPageToken(pageToken);
            // Supported attributes of files:
            // https://developers.google.com/drive/api/reference/rest/v3/files?hl=de#properties
            request.setFields("nextPageToken, " + RELEVANT_FIELDS);
            request.setQ(omnipotentMemberOfQContinuum);

            FileList filelist = request.execute();
            list.addAll(filelist.getFiles());

            pageToken = filelist.getNextPageToken();
        } while (pageToken != null);
        return list;
    }
}
