/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.dropbox;

import org.apache.commons.vfs2.FileSystem;
import org.apache.commons.vfs2.FileSystemConfigBuilder;
import org.apache.commons.vfs2.FileSystemOptions;


/**
 * The config builder for the various Dropbox configuration options.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class DbxFileSystemConfigBuilder extends FileSystemConfigBuilder {


    /**
     * Shared <code>DbxFileSystemConfigBuilder</code> instance.
     */
    protected static DbxFileSystemConfigBuilder sharedInstance;

    /**
     * Display name.
     */
    public final static String DISPLAY_NAME = DbxFileSystemConfigBuilder.class.getName() + ".DISPLAY_NAME";
    /**
     * Access token.
     */
    public final static String ACCESS_TOKEN = DbxFileSystemConfigBuilder.class.getName() + ".ACCESS_TOKEN";


    /**
     * Constructor method.
     *
     * @since 1.6
     */
    public DbxFileSystemConfigBuilder() {
        super("dropbox.");  // Static to find resources, do not use dynamic scheme here
    }

    /**
     * Provides the shared instance of the factory implementation.
     *
     * @return Shared instance
     * @since 1.6
     */
    public static synchronized DbxFileSystemConfigBuilder getSharedInstance() {

        if (sharedInstance == null) sharedInstance = new DbxFileSystemConfigBuilder();
        return sharedInstance;
    }

    /**
     * Provides the file system class to be configured by this builder.
     *
     * @return File system class
     * @since 1.6
     */
    @Override
    protected Class<? extends FileSystem> getConfigClass() {
        return DbxFileSystem.class;
    }

    /**
     * Setter for Dropbox account display name.
     *
     * @param options The options
     * @param name    The display name
     * @since 1.6
     */
    public void setAccountDisplayName(FileSystemOptions options, String name) {
        super.setParam(options, DISPLAY_NAME, name);
    }

    /**
     * Getter for Dropbox account display name.
     *
     * @param options The options
     * @return The display name, may be <code>null</code>
     * @since 1.6
     */
    public String getAccountDisplayName(FileSystemOptions options) {
        return super.getParam(options, DISPLAY_NAME);
    }

    /**
     * Setter for Dropbox access token.
     *
     * @param options The options
     * @param token   The access token
     * @since 1.6
     */
    public void setAccessToken(FileSystemOptions options, String token) {
        super.setParam(options, ACCESS_TOKEN, token);
    }

    /**
     * Getter for Dropbox access token.
     *
     * @param options The options
     * @return The access token, may be <code>null</code>
     * @since 1.6
     */
    public String getAccessToken(FileSystemOptions options) {
        return super.getParam(options, ACCESS_TOKEN);
    }
}
