/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.dropbox;

import com.dropbox.core.DbxDownloader;
import com.dropbox.core.DbxException;
import com.dropbox.core.DbxUploader;
import com.dropbox.core.v2.files.*;
import com.lf.vfslib.VFSLib;
import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.gdrive.GDriveClientWrapper;
import com.lf.vfslib.io.VFSRandomAccessContent;
import com.lf.vfslib.lang.JavaUtils;
import org.apache.commons.vfs2.*;
import org.apache.commons.vfs2.provider.AbstractFileName;
import org.apache.commons.vfs2.provider.AbstractFileObject;
import org.apache.commons.vfs2.provider.UriParser;
import org.apache.commons.vfs2.util.MonitorInputStream;
import org.apache.commons.vfs2.util.MonitorOutputStream;
import org.apache.commons.vfs2.util.RandomAccessMode;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.util.*;
import java.util.logging.Level;


/**
 * Represents a Dropbox file or folder.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class DbxFileObject extends AbstractFileObject {

    /**
     * The underlying file system.
     */
    final protected DbxFileSystem fileSystem;
    /**
     * The relative path on the Dropbox server.
     */
    protected String relPath;
    /**
     * Helper to avoid multiple refreshing.
     */
    protected boolean inRefresh = false;
    /**
     * The metadata for the underlying Dropbox entry, either <code>FileMetadata</code> or <code>FolderMetadata</code>.
     */
    protected Metadata dbxEntryMetadata = null;
    /**
     * The attributes specific for cloud file systems (e.g. "Content-Length" for proper upload).
     */
    protected Map<String, Object> attributes = null;


    /**
     * Constructor method.
     *
     * @param name       The file name
     * @param filesystem The file system
     * @throws NullPointerException If a parameter is <code>null</code>
     * @throws FileSystemException  If something goes wrong
     * @since 1.6
     */
    public DbxFileObject(AbstractFileName name, DbxFileSystem filesystem) throws FileSystemException {

        super(name, filesystem);

        // VFS2 uses URI's internally. Possible name normalizations for Dropbox are:

        /*
           Type                 Value of name                                             Relative name               relPath (normalized)        dbxEntryMetadata
           ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------
           Root folder          dropbox://johndoe@dropbox.com/                            "." or "/"                  ""                          FolderMetadata name=
           File in root         dropbox://johndoe@dropbox.com/battery_96.png              battery_96.png              /battery_96.png             FileMetadata name=battery_96.png
           Subfolder            dropbox://johndoe@dropbox.com/testfolder/                 testfolder/                 /testfolder                 FolderMetadata name=testfolder
           File in subfolder    dropbox://johndoe@dropbox.com/testfolder/check2_16.png    testfolder/check2_16.png    /testfolder/check2_16.png   FileMetadata name=check2_16.png
         */

        // See statSelf() method on how the dbxEntryMetadata is resolved via Dropbox API.

        // Remember that the "dropbox:" protocol is virtual only and cannot be queried e.g. in a browser.
        // The appkey etc. are embedded in the connection options and not part of the VFS2 URI.

        this.fileSystem = filesystem;
        this.relPath = UriParser.decode(filesystem.getRootName().getRelativeName(name));
        if (this.relPath.equals(".") || this.relPath.equals("/")) this.relPath = "";  // Does not work with Dropbox!
        else if (!this.relPath.startsWith("/")) this.relPath = '/' + this.relPath;  // Files
    }

    /**
     * Clean-up method to help the gc.
     *
     * @throws Throwable Error indication
     * @since 1.6
     */
    @Override
    protected void finalize() throws Throwable {

        super.finalize();

        if (this.attributes != null) this.attributes.clear();
    }

    /**
     * Checks if the current relative path represents the root folder.
     *
     * @return Root folder?
     * @since 2.8
     */
    protected boolean isRoot() {
        return this.relPath.isEmpty();
    }

    /**
     * Detach from the file system.
     *
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected void doDetach() throws Exception {
        this.dbxEntryMetadata = null;
    }

    /**
     * Refresh this file.
     *
     * @throws FileSystemException If something goes wrong
     * @since 1.6
     */
    @Override
    public void refresh() throws FileSystemException {

        if (!this.inRefresh) {  // Avoid redundant refreshing
            try {
                this.inRefresh = true;
                super.refresh();
            } finally {
                this.inRefresh = false;
            }
        }
    }

    /**
     * Determines the type of this file, returns <code>null</code> if the file does not exist.
     *
     * @return The file type
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected FileType doGetType() throws Exception {

        statSelf();

        // May still be null if an anticipated file is queried (e.g. abc.txt.sha)
        if (isRoot()) return FileType.FOLDER; // The root / is not supported by Dropbox API v2
        if (this.dbxEntryMetadata != null) {
            if (this.dbxEntryMetadata instanceof FileMetadata) return FileType.FILE;
            else if (this.dbxEntryMetadata instanceof FolderMetadata) return FileType.FOLDER;
        }

        FileType type = FileType.IMAGINARY; // File/folder does not exist
        try {
            // This is special here: IMAGINARY has no attributes, allow attributes for setting upload "Content-Length"
            Field field = type.getClass().getDeclaredField("hasAttrs");  // enum
            field.setAccessible(true);  // Override access limitations for private, protected etc.
            field.set(type, true);
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        return type;
    }

    /**
     * Determines the type of this file.
     *
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    protected void statSelf() throws Exception {

        if (this.dbxEntryMetadata != null) return;
        if (isRoot()) return; // The root / is not supported by Dropbox API v2
        //Thread.dumpStack();

        DbxClientWrapper client = this.fileSystem.getClient();
        try {
            this.dbxEntryMetadata = client.dbxClient.files().getMetadata(this.relPath);
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e); // Needed by developers
            this.dbxEntryMetadata = null;  // Does not exist
        } finally {
            this.fileSystem.putClient(client);
        }
    }

    /**
     * Called when the type or content of this file changes.
     *
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected void onChange() throws Exception {

        this.dbxEntryMetadata = null;
        statSelf();
    }

    /**
     * Creates this file as a folder.
     *
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected void doCreateFolder() throws Exception {

        this.statSelf();

        if (!isRoot()) {
            DbxClientWrapper client = this.fileSystem.getClient();
            try {
                CreateFolderResult result = client.dbxClient.files().createFolderV2(this.relPath);
                return;
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                this.fileSystem.putClient(client);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_CREATE_FOLDER_ERROR"));
    }

    /**
     * Gets the last modification date and time.
     *
     * @return The last modification time in milliseconds
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected long doGetLastModifiedTime() throws Exception {

        this.statSelf();

        if (this.dbxEntryMetadata instanceof FileMetadata) { // Not available for folders
            try {
                Date modtime = ((FileMetadata) this.dbxEntryMetadata).getServerModified();
                if (modtime != null) return modtime.getTime();
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_UNKNOWN_MOD_TIME_ERROR"));
    }

    /**
     * Sets the last modified time of this file (not supported by Dropbox).
     *
     * @param modtime Last modification time in milliseconds
     * @return Successful?
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected boolean doSetLastModifiedTime(long modtime) throws Exception {
        return false;  // Not supported
    }

    /**
     * Deletes the file.
     *
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected void doDelete() throws Exception {

        this.statSelf();

        if (!isRoot()) {
            DbxClientWrapper client = this.fileSystem.getClient();
            try {
                DeleteResult result = client.dbxClient.files().deleteV2(this.relPath);
                return;
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                this.fileSystem.putClient(client);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(GDriveClientWrapper.class.getName() + "_DELETE_ERROR")); // Recycled
    }

    /**
     * Rename the file.
     *
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected void doRename(FileObject newfile) throws Exception {

        this.statSelf();

        if (!isRoot()) {
            DbxClientWrapper client = this.fileSystem.getClient();
            try {
                RelocationResult result = client.dbxClient.files().moveV2(this.relPath, ((DbxFileObject) newfile).relPath);
                return;
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                this.fileSystem.putClient(client);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_RENAME_ERROR"));
    }

    /**
     * Lists the children of this file.
     *
     * @return The list of children, may be empty
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected FileObject[] doListChildrenResolved() throws Exception {

        FileObject[] array = FileObject.EMPTY_ARRAY;
        DbxClientWrapper client = this.fileSystem.getClient();
        FileSystemManager fsmanager = getFileSystem().getFileSystemManager();

        statSelf();

        try {
            ListFolderResult result = client.dbxClient.files().listFolder(this.relPath);
            List<Metadata> list = result.getEntries();

            // Convert to array with VFS files
            for (Metadata child : list) {
                FileObject fileobj = this.fileSystem.resolveFile(
                        fsmanager.resolveName(getName(), UriParser.encode(child.getName()), NameScope.CHILD));
                array = (FileObject[]) JavaUtils.addToArray(array, fileobj);

                // Populate required fields, no need to call statSelf() again
                ((DbxFileObject) fileobj).dbxEntryMetadata = child;
            }
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
            throw e;
        } finally {
            this.fileSystem.putClient(client);
        }
        return array;
    }

    /**
     * Lists the children of this file.
     *
     * @return The list of children, may be empty
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected String[] doListChildren() throws Exception {

        FileObject[] children = this.doListChildrenResolved();
        return Arrays.stream(children).map(String::valueOf).toArray(String[]::new);
    }

    /**
     * Returns the size of the file content (in bytes).
     *
     * @return The size, -1 if not available
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected long doGetContentSize() throws Exception {

        statSelf();

        if (this.dbxEntryMetadata instanceof FileMetadata) { // Not available for folders
            try {
                return ((FileMetadata) this.dbxEntryMetadata).getSize();
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_UNKNOWN_SIZE_ERROR"));
    }

    /**
     * Provides the instance to model random access for files.
     *
     * @return The instance
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected RandomAccessContent doGetRandomAccessContent(RandomAccessMode mode) throws Exception {
        return new VFSRandomAccessContent(this, mode);
    }

    /**
     * Creates an input stream to read the file content from.
     *
     * @return The input stream
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected InputStream doGetInputStream() throws Exception {

        this.statSelf();

        if (!FileType.FILE.equals(getType())) { // Required to exist
            String text = VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_READ_NOT_FILE_ERROR");
            throw new FileSystemException(text, getName());
        }

        DbxClientWrapper client = this.fileSystem.getClient();

        // VFS-113: Avoid NPE
        synchronized (this.fileSystem) {
            try {
                DbxDownloader<FileMetadata> downloader = client.dbxClient.files().download(this.relPath, null);  // Latest revision
                return new DbxInputStream(client, downloader, downloader.getInputStream());
            } finally {
            }  // For client release and stream close see DbxInputStream
        }
    }

    /**
     * Creates an output stream to write the file content to.
     *
     * @param append Not supported currently, always created freshly
     * @return The output stream
     * @throws Exception If something goes wrong
     * @since 1.6
     */
    @Override
    protected OutputStream doGetOutputStream(boolean append) throws Exception {

        this.statSelf();

        if (!FileType.FILE.equals(getType()) && !FileType.IMAGINARY.equals(getType())) {
            String text = VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_WRITE_NOT_FILE_ERROR");
            throw new FileSystemException(text, getName());
        }

        DbxClientWrapper client = this.fileSystem.getClient();

        // VFS-113: Avoid NPE
        synchronized (this.fileSystem) {
            try {
                DbxUploader<FileMetadata, UploadError, UploadErrorException> uploader = client.dbxClient.files().upload(this.relPath);
                return new DbxOutputStream(client, uploader, uploader.getOutputStream());
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                // For client release and stream close see DbxOutputStream
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_OUTPUTSTREAM_ERROR"));
    }

    /**
     * An <code>InputStream</code> that monitors for end-of-file.
     *
     * @since 1.6
     */
    protected class DbxInputStream extends MonitorInputStream {

        protected DbxClientWrapper client;
        protected DbxDownloader<FileMetadata> downloader;

        public DbxInputStream(DbxClientWrapper client, DbxDownloader<FileMetadata> downloader, InputStream in) {

            super(in);
            this.client = client;
            this.downloader = downloader;
        }

        @Override
        protected void onClose() throws IOException {
            try {
                this.downloader.close();
            } finally {
                fileSystem.putClient(this.client);
            }
        }
    }

    /**
     * An <code>OutputStream</code> that wraps a Dropbox <code>OutputStream</code>, and closes it when the stream is closed.
     *
     * @since 1.6
     */
    protected class DbxOutputStream extends MonitorOutputStream {


        protected DbxClientWrapper client;
        protected DbxUploader<FileMetadata, UploadError, UploadErrorException> uploader = null;

        public DbxOutputStream(DbxClientWrapper client, DbxUploader<FileMetadata, UploadError, UploadErrorException> uploader,
                               OutputStream out) {

            super(out);
            this.client = client;
            this.uploader = uploader;
        }

        @Override
        protected void onClose() throws IOException {
            try {
                this.uploader.finish();
            } catch (DbxException dbexc) {
                throw new IOException(dbexc);
            } finally {
                try {
                    this.uploader.close();
                } catch (Exception ignored) {
                }
                fileSystem.putClient(this.client);
            }
        }
    }

    /**
     * Provides the VFSLib instance for this filesystem.
     *
     * @return The instance
     * @since 1.6
     */
    public VFSLib getVFSLib() {
        return this.fileSystem.getVFSLib();
    }

    /**
     * Provides the <code>ClientWrapper</code> implementation for this file system.
     *
     * @return The client wrapper holding the low-level client
     * @throws IOException If an I/O error occurs
     * @since 2.8
     */
    public DbxClientWrapper getDbxClientWrapper() throws IOException {
        return this.fileSystem.getClient();
    }

    /**
     * Returns the attributes specific for cloud file systems (e.g. "Content-Length" for proper upload)
     *
     * @return The map
     * @since 1.6
     */
    @Override
    protected Map<String, Object> doGetAttributes() throws Exception {

        synchronized (this) {
            if (this.attributes == null) this.attributes = new HashMap<>(0);
        }
        return this.attributes;
    }

    /**
     * Sets an attribute of this cloud file.
     *
     * @param attrName The name (ATTR_CONTENT_LENGTH|...)
     * @param value    The value
     * @since 1.6
     */
    @Override
    protected void doSetAttribute(String attrName, Object value) throws Exception {

        doGetAttributes();  // Make sure cache exists
        this.attributes.put(attrName, value);
    }
}
