/*
    Copyright (c) 2005-2025 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.core;

import com.lf.vfslib.lang.StringUtils;
import com.lf.vfslib.net.VFSFileSystemManager;
import com.lf.vfslib.resource.ResourceLocator;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * Global settings for the VFSLib package.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class VFSLibSettings {


    /**
     * Shared <code>VFSLibSettings</code> instance.
     */
    protected static VFSLibSettings sharedInstance;
    /**
     * Shared <code>FileSystemManager</code> instance holding the network connections.
     */
    protected static VFSFileSystemManager vfsManagerNetwork;
    /**
     * Shared <code>FileSystemManager</code> instance for any network protocol including file://.
     */
    protected static VFSFileSystemManager vfsManagerAllProtocols;

    /**
     * Prefix used for the log messages like "VFSLIB-0001".
     */
    public final static String MSG_PREFIX = "VFSLIB";
    /**
     * Prefix used for the log messages created by this module.
     */
    public final static String LOG_PREFIX = "[vfslib] ";

    /**
     * Locale to use for the messages presented to the user (default is English).
     */
    protected Locale userLocale;
    /**
     * Manager object for language specific texts.
     */
    protected ResourceBundle texts;
    /**
     * Manager object for language specific message codes like "VFSLIB-0001" and texts.
     */
    protected ResourceBundle mcodes;
    /**
     * Locale to use for the logging and debugging messages (default is English).
     */
    protected final Locale logLocale;
    /**
     * Application wide logging instance (default is logging disabled).
     */
    protected Logger logger;
    /**
     * Holds deploy properties like the exact version and build.
     */
    protected final Properties version;


    /**
     * Constructor method.
     *
     * @since 1.6
     */
    public VFSLibSettings() {

        if (VFSLibSettings.sharedInstance == null) VFSLibSettings.sharedInstance = this;

        // Disable annoying VFS log messages, enable later
        System.setProperty("org.apache.commons.logging.Log", "org.apache.commons.logging.impl.NoOpLog");

        this.logLocale = Locale.ENGLISH;
        this.version = new Properties();
        try {
            this.version.load(ResourceLocator.class.getResourceAsStream("deploy.properties"));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        Logger nologger = Logger.getLogger(this.getClass().getName());
        nologger.setLevel(Level.OFF);  // Disabled by default
        this.logger = nologger;

        // Set the default user locale (currently English). Applications may call the setUserLocale() method
        // again in order to set a locale e,g, from command line arguments.
        try {
            this.setUserLocale(Locale.ENGLISH);
        } catch (Exception ignored) {
        }
        try {
            // Create localized texts
            this.texts = ResourceBundle.getBundle(VFSLibSettings.class.getName() + "_texts");
        } catch (Exception ignored) {
        }
        try {
            // Create message codes like "VFSLIB-0001"
            this.mcodes = ResourceBundle.getBundle(VFSLibSettings.class.getName() + "_mcodes");
        } catch (Exception ignored) {
        }
    }

    /**
     * Get a shared instance of the Commons <code>FileSystemManager</code> holding the network connections.
     * <p>
     * Here, the connections are managed while the global <code>VFSFileSystemManager</code> does not.
     * The manager provides support for "real" network protocols like "http" or "sftp", all file based
     * protocols like "jar" and "file" are automatically removed.
     *
     * @return Shared instance
     * @since 1.6
     */
    public static synchronized VFSFileSystemManager getManagerNetwork() {

        if (vfsManagerNetwork == null) {
            // Setup our local VFS file system manager used by the VFSSessionWindow
            vfsManagerNetwork = new VFSFileSystemManager();
            String[] ignored = new String[]{"ram", "res", "tmp", "ear", "jar", "par", "sar", "war",
                    "ejb3", "bz2", "file", "gz", "tar", "tbz2", "tgz", "zip"};
            for (String protocol : ignored) {
                vfsManagerNetwork.removeOperationProvider(protocol);
            }
        }
        return vfsManagerNetwork;
    }

    /**
     * Get a shared instance of the Commons <code>FileSystemManager</code> with support for all protocols.
     * <p>
     * This one should be used when accessing files/folders on the classpath either "jar": or "file:".
     *
     * @return New instance
     * @since 2.8
     */
    public static synchronized VFSFileSystemManager getManagerAllProtocols() {

        if (vfsManagerAllProtocols == null) {
            vfsManagerAllProtocols = new VFSFileSystemManager();
        }
        return vfsManagerAllProtocols;
    }

    /**
     * Performs some simple tests for the caches and configurations and creates log messages.
     * <p>
     * If either the configurations, or the message codes, or the build properties are absent,
     * then the application should be interrupted since this is fatal.
     *
     * @since 1.6
     */
    public void check() {

        // The messages are hard-coded since the message code loading could fail

        if (this.texts != null) {
            this.logger.log(Level.CONFIG, LOG_PREFIX + "Internationalized texts successfully loaded");
        } else {
            logger.log(Level.SEVERE, MSG_PREFIX + "-0005: Internationalized texts could not be found");
        }

        String test = getUserMessage(MSG_PREFIX + "-0000");
        if (test != null) {
            this.logger.log(Level.CONFIG, LOG_PREFIX + "Message codes successfully loaded");
        } else this.logger.log(Level.SEVERE, MSG_PREFIX + "-0002: Message codes not available");

        if (this.version.getProperty("VERSION") != null) {
            this.logger.log(Level.CONFIG, LOG_PREFIX + "Build properties successfully loaded");
        } else this.logger.log(Level.SEVERE, MSG_PREFIX + "-0003: Build properties not available");
    }

    /**
     * Gets the global logging instance.
     *
     * @return Logging instance
     * @since 1.6
     */
    public Logger getLogger() {
        return this.logger;
    }

    /**
     * Sets the global logging instance.
     *
     * @param logger Logging instance
     * @throws NullPointerException Error indication
     * @since 1.6
     */
    public void setLogger(Logger logger) {

        if (logger != null) {
            this.logger = logger;
            this.logger.log(Level.CONFIG, LOG_PREFIX + "Setting new logger with log level " + logger.getLevel());
        } else throw new NullPointerException("Invalid logger");
    }

    /**
     * Gets the global user locale.
     *
     * @return Current user locale
     * @since 1.6
     */
    public Locale getUserLocale() {
        return this.userLocale;
    }

    /**
     * Sets the global user locale.
     * <p>
     * Make sure that the given new locale is supported by the settings class.
     *
     * @param locale Locale instance
     * @throws IllegalArgumentException Error indication
     * @since 1.6
     */
    public void setUserLocale(Locale locale) throws IllegalArgumentException {

        if (locale.getLanguage().equals("en")) Locale.setDefault(Locale.ENGLISH);
        else throw new IllegalArgumentException("Locale " + locale + " is currently not supported");
        this.userLocale = Locale.getDefault();
    }

    /**
     * Gets the global locale used for logging.
     * <p>
     * At the moment this is static since exceptions and most System.outs are in English.
     *
     * @return Current logging locale
     * @since 1.6
     */
    public Locale getLogLocale() {
        return this.logLocale;
    }

    /**
     * Gets the global version information.
     *
     * @return Current version information
     * @since 1.6
     */
    public Properties getDeployProps() {
        return this.version;
    }

    /**
     * Provides the locale specific release notes (<code>core</code> package).
     *
     * @return The release notes as binary data, <code>null</code> if an error occurred
     * @since 2.8
     */
    public byte[] getReleaseNotes() {
        byte[] buffer = new byte[1024];
        int len;

        try {
            ByteArrayOutputStream ostream = new ByteArrayOutputStream();
            InputStream istream = ResourceLocator.class.getResourceAsStream("release_notes.xml");
            while ((len = istream.read(buffer)) != -1) {
                ostream.write(buffer, 0, len);
            }
            ostream.flush();
            return ostream.toByteArray();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Convenience method to get a shared instance.
     *
     * @return Shared instance
     * @since 1.6
     */
    public static synchronized VFSLibSettings getSharedInstance() {

        if (sharedInstance == null) sharedInstance = new VFSLibSettings();
        return sharedInstance;
    }

    /**
     * Convenience method to log a message.
     * <p>
     * Make sure that the message applies to the current log locale (normally English).
     * The log prefix like "[vfslib]" is prepended.
     *
     * @param level   The log level
     * @param message The log message
     * @since 1.6
     */
    public static void log(Level level, String message) {

        VFSLibSettings settings = VFSLibSettings.getSharedInstance();
        settings.getLogger().log(level, VFSLibSettings.LOG_PREFIX + message);
    }

    /**
     * Convenience method to log an exception.
     * <p>
     * Make sure that the message applies to the current log locale (normally English).
     * The log prefix like "[vfslib]" is prepended.
     *
     * @param level     The log level
     * @param throwable The exception
     * @since 2.8
     */
    public static void log(Level level, Throwable throwable) {

        VFSLibSettings settings = VFSLibSettings.getSharedInstance();
        settings.getLogger().log(level, VFSLibSettings.LOG_PREFIX, throwable);
    }

    /**
     * Convenience method to log a message.
     * <p>
     * Make sure that the message applies to the current log locale (normally English).
     * The log prefix like "[vfslib]" is prepended.
     *
     * @param level   The log level
     * @param message The log message
     * @since 1.6
     */
    public static void logMessage(Level level, String message) {

        VFSLibSettings settings = VFSLibSettings.getSharedInstance();
        settings.getLogger().log(level, VFSLibSettings.LOG_PREFIX + message);
    }

    /**
     * Convenience method to log a localized official message.
     * <p>
     * The message is logged using the current log locale (normally English). The domain identifier
     * is printed (e.g. "VFSLIB-0006") while the package log prefix is omitted (e.g. "[vfslib]").
     *
     * @param code  The log message id like "VFSLIB-0006"
     * @param level The log level
     * @since 1.6
     */
    public static void logMessage(String code, Level level) {

        VFSLibSettings settings = VFSLibSettings.getSharedInstance();
        String message = getUserMessage(code);
        settings.getLogger().log(level, message);
    }

    /**
     * Convenience method to get a localized official message for the user.
     * <p>
     * The message is returned using the current user locale. The domain identifier
     * is prepended (e.g. "VFSLIB-0006").
     *
     * @param code The message code of interest, don't pass <code>null</code> values
     * @return The message text, <code>null</code> if code does not exist
     * @since 1.6
     */
    public static String getUserMessage(String code) {
        try {
            VFSLibSettings settings = VFSLibSettings.getSharedInstance();
            return settings.mcodes.getString(code);
        } catch (MissingResourceException ignored) {
        }
        return null;
    }

    /**
     * Convenience method to get a localized text for the user.
     * <p>
     * The message is returned using the current user locale.
     *
     * @param variable The variable, don't pass <code>null</code> values
     * @return The localized text or the variable
     * @since 1.6
     */
    public static String getUserText(String variable) {

        String value = null;
        try {
            VFSLibSettings settings = VFSLibSettings.getSharedInstance();
            value = settings.texts.getString(variable);
        } catch (MissingResourceException ignored) {
        }
        return value != null ? value : variable;
    }

    /**
     * Convenience method to get a localized text for the user.
     * <p>
     * The message is returned using the current user locale (e.g. "eng_US").
     *
     * @param variable    The variable without class prefix, don't pass <code>null</code> values
     * @param regex       The regular expression
     * @param replacement The replacement string
     * @return The localized text or the variable
     * @since 1.6
     */
    public static String getUserText(String variable, String[] regex, String[] replacement) {

        try {
            String value = VFSLibSettings.getUserText(variable);  // Propagate
            if (value != null) {
                for (int i = 0; i < regex.length; i++) {
                    value = StringUtils.safeReplaceAll(value, regex[i], replacement[i]);
                }
                return value;
            }
        } catch (Exception e) {
            //e.printStackTrace();
        }
        return variable;
    }
}
