/*
    Copyright (c) 2005-2013 Leisenfels. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.test.s3;

import com.lf.commons.i18n.Locale;
import com.lf.vfslib.VFSLib;
import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.s3.S3FileProvider;
import com.lf.vfslib.s3.S3FileSystemConfigBuilder;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.FileType;
import org.apache.commons.vfs2.VFS;
import org.apache.commons.vfs2.impl.DefaultFileSystemManager;

import javax.swing.*;
import javax.swing.filechooser.FileNameExtensionFilter;
import java.awt.*;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;


/**
 * Example class showing the use of the Amazon&reg; S3 provider.
 * <p/>
 * Required libraries in classpath:
 * <code>
 * commons-vfs-2.0.jar          Original Apache VFS 2.0 library
 * commons-logging-1.1.jar      Required for VFS 2.0
 * aws-java-sdk-1.6.1.jar       Amazon&reg; S3 SDK
 * </code>
 * <p/>
 * @author Axel Schwolow
 * @version $Revision: 1.3 $, $Date: 2013/10/19 16:40:20 $
 * @since 1.6
 */
public class ExampleAmazonS3 {


    /**
     * Constructor method.
     * <p/>
     * @param language    The language (en|de)
     * @param licensefile The license key file
     * @param accesskeyid The access key ID from Amazon&reg;
     * @param secretkey   The secret key from Amazon&reg;
     * @param bucket      The bucket name to use
     * @throws org.apache.commons.vfs2.FileSystemException
     *          Error indication
     * @since 1.6
     */
    public ExampleAmazonS3 (String language, File licensefile, String accesskeyid, String secretkey,
                            String bucket) throws FileSystemException {

        System.out.println("Configuring VFSLib Amazon S3 provider:");
        System.out.println("    language           " + (language != null ? language : "(using English)"));
        System.out.println("    licensefile        " + licensefile);
        System.out.println("    accesskeyid        " + accesskeyid);
        System.out.println("    secretkey          " + (secretkey != null ? secretkey.replaceFirst("^.{5}", "*****") : "(not available)"));
        System.out.println("    bucket             " + bucket);

        // Messages from VFSLib should be printed in the user's language (optional, default is "eng_US").
        // VFSLib supports ISO 639-2 codes (three language letters) which allows languages like "nds" (Low German).
        VFSLibSettings settings = VFSLibSettings.getSharedInstance();
        if (language == null) language = "en";
        if (!language.equals("en")) {
            settings.setUserLocale(Locale.parseLocale(language));  // Maps "de" -> "ger_DE"
        }

        // Setup the main VFSLib instance, get evaluation license key from Leisenfels website
        VFSLib vfslib = new VFSLib();
        if (licensefile == null) {
            JFileChooser chooser = new JFileChooser("Please choose a VFSLib license file");
            chooser.setFileFilter(new FileNameExtensionFilter("License Files", "txt"));
            chooser.showOpenDialog(null);
            licensefile = chooser.getSelectedFile();
            System.out.println("The VFSLib license file is: " + licensefile);
        }
        vfslib.setLicenseFile(licensefile);          // Import from file

        // We use the default file system manager here
        DefaultFileSystemManager fsmanager = (DefaultFileSystemManager) VFS.getManager();

        // Add Amazon S3 provider to VFS, ask user for missing arguments
        if (accesskeyid == null) {
            accesskeyid = JOptionPane.showInputDialog("Please enter the access key ID from Amazon");
            System.out.println("The Amazon S3 access key ID is: " + accesskeyid);
        }
        if (secretkey == null) {
            secretkey = JOptionPane.showInputDialog("Please enter the secret key from Amazon");
            System.out.println("The Amazon S3 secret key is: " + secretkey);
        }
        S3FileProvider provider = vfslib.addProviderAmazonS3(fsmanager);
        if (provider == null) {
            System.err.println("Sorry, the Amazon S3 provider could not be activated, exiting");
            return;
        }

        String scheme = vfslib.getSchemeAmazonS3();
        System.out.println("\nAdded Amazon S3 scheme \"" + scheme + '\"');

        if (bucket == null) {
            bucket = JOptionPane.showInputDialog("Please enter the name of the Amazon S3 bucket");
            System.out.println("The Amazon S3 bucket is: " + bucket);
        }

        // Pass bucket name over to VFSLib to access the Amazon S3 account
        FileSystemOptions options = new FileSystemOptions();
        S3FileSystemConfigBuilder builder = new S3FileSystemConfigBuilder();
        builder.setBucketName(options, bucket);

        // Setup proper account name, used as user name for Amazon S3 URLs (failsafe is "anonymous"):
        //    s3://[name]:[secret key]@aws.amazon.com
        String username = "johndoe";
        builder.setAccountDisplayName(options, username);
        builder.setAccessKeyID(options, accesskeyid);
        builder.setSecretKey(options, secretkey);
        builder.setBucketName(options, bucket);


        // List a Amazon S3 folder
        String uri = scheme + "://" + username + "@aws.amazon.com";
        System.out.println("Listing child entries of " + uri + ":");
        try {
            FileObject fileobj = fsmanager.resolveFile(uri, options);
            if (fileobj.getType().equals(FileType.FOLDER)) {
                FileObject[] childs = fileobj.getChildren();
                for (FileObject next : childs) {
                    if (next.getType().equals(FileType.FOLDER)) {
                        System.out.println("    DIR:  " + String.valueOf(next));
                    }
                    else System.out.println("    FILE: " + String.valueOf(next));
                }
            }
            else System.out.println("    Entry " + uri + " is not a folder");
        }
        catch (Exception e) { e.printStackTrace(); }

        // The following code modifies files in the Amazon S3 file system.
        // Please uncomment the lines and adjust values for your needs.

        // Create a Amazon S3 folder
        String tempfolder = scheme + "://" + username + "@aws.amazon.com/0123456789abcdefghijklmnopqrstuvwxyz";
        System.out.println("Creating temporary folder " + tempfolder + ":");
        boolean success = false;
        try {
            FileObject fileobj = fsmanager.resolveFile(tempfolder, options);
            if (!fileobj.exists()) {
                fileobj.createFolder();
                success = fileobj.exists();
                System.out.println(success ? "    Successful" : "    Failed");
            }
            else {
                System.out.println("    Entry " + tempfolder + " does already exist, exiting");
                return;
            }
        }
        catch (Exception e) { e.printStackTrace(); }
        if (!success) {
            System.err.println("Sorry, could not create folder, exiting");
            return;
        }

        String content = "Hello world!";
        String encoding = "ISO-8859-1";

        // Upload a file to Amazon S3. Get file type, last modified, and size.
        String tempfile = tempfolder + "/readme.txt";
        System.out.println("Uploading temporary file " + tempfile + ":");
        success = false;
        try {
            FileObject fileobj = fsmanager.resolveFile(tempfile, options);
            OutputStream ostream = fileobj.getContent().getOutputStream();
            ostream.write(content.getBytes(encoding));
            ostream.flush();
            ostream.close();

            success = fileobj.exists();
            System.out.println(success ? "    Successful" : "    Failed");

            System.out.println("    TYPE:  " + fileobj.getType().getName());
            System.out.println("    MOD:   " + new Date(fileobj.getContent().getLastModifiedTime()));  // Currently not supported for folders
            System.out.println("    SIZE:  " + fileobj.getContent().getSize());
        }
        catch (Exception e) { e.printStackTrace(); }
        if (!success) {
            System.err.println("Sorry, could not upload file, exiting");
            return;
        }

        // Rename a S3 file
        String tempfilerenamed = tempfolder + "/README";  // README.TXT not possible currently
        System.out.println("Renaming temporary file " + tempfile + " to " + tempfilerenamed + ":");
        success = false;
        try {
            FileObject fileobj = fsmanager.resolveFile(tempfile, options);
            FileObject fileobjrenamed = fsmanager.resolveFile(tempfilerenamed, options);
            fileobj.moveTo(fileobjrenamed);
            success = fileobjrenamed.exists();
            System.out.println(success ? "    Successful" : "    Failed");
            if (success) tempfile = tempfilerenamed;
        }
        catch (Exception e) { e.printStackTrace(); }
        if (!success) {
            System.err.println("Sorry, could not rename file, exiting");
            return;
        }

        // Download a file from S3
        System.out.println("Downloading temporary file " + tempfile + ":");
        success = false;
        try {
            FileObject fileobj = fsmanager.resolveFile(tempfile, options);
            ByteArrayOutputStream bostream = new ByteArrayOutputStream();
            InputStream istream = fileobj.getContent().getInputStream();
            int len;
            byte[] buffer = new byte[1024];
            while ((len = istream.read(buffer)) != -1) {
                bostream.write(buffer, 0, len);
            }
            istream.close();
            bostream.flush();
            bostream.close();
            String loaded = new String(bostream.toByteArray(), encoding);

            success = loaded.equals(content);
            System.out.println(success ? "    Successful (content=" + loaded + ")" : "    Failed");
        }
        catch (Exception e) { e.printStackTrace(); }
        if (!success) {
            System.err.println("Sorry, could not download file, exiting");
            return;
        }

        // Remove a S3 file and folder
        System.out.println("Removing temporary entries:");
        success = false;
        try {
            // Remove file
            FileObject fileobj = fsmanager.resolveFile(tempfile, options);
            System.out.print("    " + tempfile);
            success = fileobj.delete();
            System.out.println(success ? "  Successful" : "    Failed");

            // Remove folder
            fileobj = fsmanager.resolveFile(tempfolder, options);
            System.out.print("    " + tempfolder);
            success = fileobj.delete();
            System.out.println(success ? "  Successful" : "    Failed");
        }
        catch (Exception e) { e.printStackTrace(); }
        if (!success) {
            System.err.println("Sorry, could not remove file/folder, exiting");
            return;
        }
    }

    /**
     * Functionality for testing and debugging.
     * <p/>
     * Supported arguments:
     * <code>
     * -language [value]           Language "en" or "de"
     * -licensefile [file]         License file
     * -accesskeyid [value]        Your access key ID from Amazon&reg;
     * -secretkey [value]          Your secret key from Amazon&reg;
     * -bucket [value]             The bucket name to use
     * </code>
     * <p/>
     * @param args Array of strings with console arguments
     * @since 1.6
     */
    public static void main (String[] args) {

        String language = null, accesskeyid = null, secretkey = null, bucket = null;
        File licensefile = null;

        try {
            if (GraphicsEnvironment.isHeadless()) {
                System.err.println("\nSorry, this example requires a graphical environment, exiting");
                System.exit(1);
            }

            // Disable annoying VFS log messages like:
            // 20.09.2013 13:48:31 org.apache.commons.vfs2.VfsLog info
            // INFO: Using "C:\DOCUME~1\User1\LOCALS~1\Temp\vfs_cache" as temporary files store.
            System.setProperty("org.apache.commons.logging.Log", "org.apache.commons.logging.impl.NoOpLog");
            System.out.println("");

            // Parse arguments
            for (int i = 0; i < args.length; i++) {
                if (args[i].equals("-language") && (i + 1) < args.length) language = args[++i];
                else if (args[i].equals("-licensefile") && (i + 1) < args.length) licensefile = new File(args[++i]);
                else if (args[i].equals("-accesskeyid") && (i + 1) < args.length) accesskeyid = args[++i];
                else if (args[i].equals("-secretkey") && (i + 1) < args.length) secretkey = args[++i];
                else if (args[i].equals("-bucket") && (i + 1) < args.length) bucket = args[++i];
            }

            new ExampleAmazonS3(language, licensefile, accesskeyid, secretkey, bucket);
            System.exit(0);
        }
        catch (Exception exc) {
            try { Thread.sleep(1000); }
            catch (Exception e) { }
            exc.printStackTrace();
        }
        System.exit(1);
    }
}
