/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.s3;

import com.lf.vfslib.VFSLib;
import org.apache.commons.vfs2.*;
import org.apache.commons.vfs2.provider.AbstractOriginatingFileProvider;
import org.apache.commons.vfs2.provider.GenericFileName;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;


/**
 * A provider to access the files hosted by Amazon S3.
 * <p>
 * This class coordinates the whole Amazon S3 functionality used by VFS2.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class S3FileProvider extends AbstractOriginatingFileProvider {


    /**
     * The parental object.
     */
    protected final VFSLib vfsLib;


    /**
     * The provider's capabilities.
     */
    static final Collection<Capability> capabilities =
            Collections.unmodifiableCollection(Arrays.asList(
                    //Capability.APPEND_CONTENT,
                    //Capability.ATTRIBUTES,
                    //Capability.COMPRESS,
                    Capability.CREATE,
                    Capability.DELETE,
                    //Capability.DIRECTORY_READ_CONTENT,
                    //Capability.DISPATCHER,
                    //Capability.FS_ATTRIBUTES,
                    Capability.GET_LAST_MODIFIED,
                    Capability.GET_TYPE,
                    //Capability.JUNCTIONS,
                    //Capability.LAST_MODIFIED,
                    Capability.LIST_CHILDREN,
                    //Capability.MANIFEST_ATTRIBUTES,
                    Capability.RANDOM_ACCESS_READ,
                    //Capability.RANDOM_ACCESS_WRITE,
                    Capability.READ_CONTENT,
                    //Capability.RENAME,
                    //Capability.SET_LAST_MODIFIED_FILE,
                    //Capability.SET_LAST_MODIFIED_FOLDER,
                    //Capability.SIGNING,
                    Capability.URI,
                    //Capability.VIRTUAL,
                    Capability.WRITE_CONTENT));


    /**
     * Constructor method (do not use).
     *
     * @throws InstantiationException Error indication
     */
    public S3FileProvider() throws InstantiationException {
        throw new InstantiationException("Do not use this constructor!");
    }

    /**
     * Constructor method.
     *
     * @param vfslib The parental object
     * @throws NullPointerException     If parameters are <code>null</code>
     * @throws IllegalArgumentException If an error occurs
     */
    public S3FileProvider(VFSLib vfslib) {

        super();

        if (vfslib == null) throw new NullPointerException();

        // Using this info an authorize URL can be acquired from Amazon
        // in order to let the user allow this application to access his Amazon resources.
        this.vfsLib = vfslib;

        setFileNameParser(S3FileNameParser.getSharedInstance());
    }

    /**
     * Creates a Amazon file system.
     *
     * @param name    The file name
     * @param options The file system options
     * @throws org.apache.commons.vfs2.FileSystemException If something goes wrong
     */
    @Override
    protected FileSystem doCreateFileSystem(FileName name, FileSystemOptions options) throws FileSystemException {

        // Create the file system
        GenericFileName rootName = (GenericFileName) name;

        // Check out for a free connection here to detect errors
        S3ClientWrapper client = S3ClientWrapper.getS3ClientWrapper(rootName, options, this.vfsLib, this);
        S3ClientWrapper.unlockS3ClientWrapper(client);

        return new S3FileSystem(rootName, options, this.vfsLib, this);
    }

    /**
     * Provides the builder instance responsible for proper file system options.
     *
     * @return The builder
     */
    @Override
    public FileSystemConfigBuilder getConfigBuilder() {
        return S3FileSystemConfigBuilder.getSharedInstance();
    }

    /**
     * Provides the provider's capabilities.
     *
     * @return The collection
     */
    public Collection<Capability> getCapabilities() {
        return S3FileProvider.capabilities;
    }

    /**
     * Provides the parental manager object.
     *
     * @return The manager
     * @since 2.8
     */
    public VFSLib getVFSLib() {
        return vfsLib;
    }
}
