/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.s3;

import com.lf.vfslib.VFSLib;
import com.lf.vfslib.core.VFSLibConstants;
import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.dropbox.DbxClientWrapper;
import com.lf.vfslib.gdrive.GDriveClientWrapper;
import com.lf.vfslib.io.VFSRandomAccessContent;
import com.lf.vfslib.lang.JavaUtils;
import org.apache.commons.vfs2.*;
import org.apache.commons.vfs2.provider.AbstractFileName;
import org.apache.commons.vfs2.provider.AbstractFileObject;
import org.apache.commons.vfs2.provider.UriParser;
import org.apache.commons.vfs2.util.MonitorInputStream;
import org.apache.commons.vfs2.util.MonitorOutputStream;
import org.apache.commons.vfs2.util.RandomAccessMode;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.model.*;

import java.io.*;
import java.lang.reflect.Field;
import java.time.Instant;
import java.util.*;
import java.util.logging.Level;


/**
 * Represents a Amazon S3 file or folder.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class S3FileObject extends AbstractFileObject {

    /**
     * The key used for the root folder (bucket).
     */
    final static protected String ROOT_KEY = "/";

    /**
     * The underlying file system.
     */
    final protected S3FileSystem fileSystem;
    /**
     * The relative path on the Amazon S3 server.
     */
    protected String relPath;
    /**
     * Helper to avoid multiple refreshing.
     */
    protected boolean inRefresh = false;
    /**
     * The underlying Amazon S3 file (either file or <code>s3Folder</code> is set).
     */
    protected S3Object s3Object = null;
    /**
     * The underlying Amazon S3 folder (either folder or <code>s3Object</code> is set).
     */
    protected CommonPrefix s3Folder = null;
    /**
     * The attributes specific for cloud file systems (e.g. "Content-Length" for proper upload).
     */
    protected Map<String, Object> attributes = null;


    /**
     * Constructor method.
     *
     * @param name       The file name
     * @param filesystem The file system
     * @throws NullPointerException                        If a parameter is <code>null</code>
     * @throws org.apache.commons.vfs2.FileSystemException If something goes wrong
     */
    public S3FileObject(AbstractFileName name, S3FileSystem filesystem) throws FileSystemException {

        super(name, filesystem);

        // VFS2 uses URI's internally. Possible name normalizations for Amazon S3 are:

        /*
           Type                 Value of name                                           Relative name               relPath (normalized)        s3Object                        s3Folder
           ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
           Root folder          s3://johndoe@aws.amazon.com/                            "." or "/"                  "/"                         null                            prefix=/
           File in root         s3://johndoe@aws.amazon.com/battery_96.png              battery_96.png              battery_96.png              key=battery_96.png              null
           Subfolder            s3://johndoe@aws.amazon.com/testfolder/                 testfolder/                 testfolder                  null                            prefix=testfolder/
           File in subfolder    s3://johndoe@aws.amazon.com/testfolder/check2_16.png    testfolder/check2_16.png    testfolder/check2_16.png    key=testfolder/check2_16.png    null
         */

        // See statSelf() method on how the s3Object or s3Folder is resolved via AWS API.

        // Remember that the "s3:" protocol is virtual only and cannot be queried e.g. in a browser.
        // The bucket name, region etc. are embedded in the connection options and not part of the VFS2 URI.

        this.fileSystem = filesystem;
        this.relPath = UriParser.decode(filesystem.getRootName().getRelativeName(name));
        if (this.relPath.equals(".") || isRoot()) this.relPath = ROOT_KEY;  // Root (bucket)
        else {
            this.relPath = this.relPath.replaceAll("^([/]+)", ""); // Remove leading/trailing slashes
            this.relPath = this.relPath.replaceAll("([/]+)$", "");
        }
    }

    /**
     * Clean-up method to help the gc.
     *
     * @throws Throwable Error indication
     */
    @Override
    protected void finalize() throws Throwable {

        super.finalize();

        if (this.attributes != null) this.attributes.clear();
    }

    /**
     * Checks if the current relative path represents the root folder (bucket).
     *
     * @return Root folder?
     * @since 2.8
     */
    protected boolean isRoot() {
        return this.relPath.equals(ROOT_KEY);
    }

    /**
     * Provides the Amazon S3 bucket name.
     *
     * @return Bucket, <code>null</code> if not available
     */
    protected String getS3BucketName() {
        try {
            FileSystemOptions fsoptions = this.fileSystem.getFileSystemOptions();
            return S3FileSystemConfigBuilder.getSharedInstance().getBucketName(fsoptions);
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Detach from the file system.
     *
     * @throws Exception If something goes wrong
     */
    @Override
    protected void doDetach() throws Exception {
        this.s3Object = null;
    }

    /**
     * Refresh this file.
     *
     * @throws org.apache.commons.vfs2.FileSystemException If something goes wrong
     */
    @Override
    public void refresh() throws FileSystemException {

        if (!this.inRefresh) {  // Avoid redundant refreshing
            try {
                this.inRefresh = true;
                super.refresh();
            } finally {
                this.inRefresh = false;
            }
        }
    }

    /**
     * Determines the type of this file, returns <code>null</code> if the file does not exist.
     *
     * @return The file type
     * @throws Exception If something goes wrong
     */
    @Override
    protected FileType doGetType() throws Exception {

        statSelf();

        if (isRoot()) return FileType.FOLDER;
        else if (this.s3Object != null) { // Resolved
            // May still be null if an anticipated file is queried (e.g. abc.txt.sha)
            return FileType.FILE;
        } else if (this.s3Folder != null) { // Resolved
            return FileType.FOLDER;
        }
        FileType type = FileType.IMAGINARY; // File/folder does not exist
        try {
            // This is special here: IMAGINARY has no attributes, allow attributes for setting upload "Content-Length"
            Field field = type.getClass().getDeclaredField("hasAttrs");  // enum
            field.setAccessible(true);  // Override access limitations for private, protected etc.
            field.set(type, true);
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        return type;
    }

    /**
     * Determines the type of this file.
     *
     * @throws Exception If something goes wrong
     */
    protected void statSelf() throws Exception {

        if (this.s3Object != null || this.s3Folder != null || isRoot()) return;  // Already resolved or root

        S3ClientWrapper client = this.fileSystem.getClient();
        try {
            // Must be compatible with folders created with AWS console.
            // Found here: http://stackoverflow.com/questions/1939743/amazon-s3-boto-how-to-create-folder

            // relPath has been normalized to never carry a trailing "/", add now to evaluate folder properly
            String relpathfile = (this.relPath.endsWith("/") ? this.relPath : this.relPath.replaceAll("([/]+)$", "")); // Remove /
            String relpathfolder = (this.relPath.endsWith("/") ? this.relPath : this.relPath + '/'); // Add /

            // GetObjectRequest does not provide the S3Object (!), list parent folder instead
            ListObjectsV2Response listresponse;
            if (!this.relPath.contains("/")) { // Child of root
                ListObjectsV2Request listrequest = ListObjectsV2Request.builder() //
                        .bucket(getS3BucketName()) //
                        .delimiter("/")
                        .build();
                listresponse = client.s3Client.listObjectsV2(listrequest);
            } else { // Child of subfolder
                ListObjectsV2Request listrequest = ListObjectsV2Request.builder() //
                        .bucket(getS3BucketName()) //
                        .prefix(this.relPath) //
                        .delimiter("/")
                        .build();
                listresponse = client.s3Client.listObjectsV2(listrequest);
            }

            // 1. step: evaluate folders (prefixes here)
            List<CommonPrefix> folders = listresponse.commonPrefixes();
            for (CommonPrefix nextfolder : folders) {
                if (nextfolder.prefix().equals(relpathfolder)) { // testfolder/ or testfolder/testtestfolder/
                    this.s3Object = null;
                    this.s3Folder = nextfolder;
                    return;
                }
            }

            // 2. step: evaluate files (objects here)
            List<S3Object> children = listresponse.contents();
            for (S3Object child : children) {
                if (child.key().equals(relpathfile)) { // battery_96.png or testfolder/check2_16.png
                    this.s3Object = child;
                    this.s3Folder = null;
                    return;
                }
            }
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);  // Needed by developers
            this.s3Object = null;
        } finally {
            this.fileSystem.putClient(client);
        }
    }

    /**
     * Called when the type or content of this file changes.
     *
     * @throws Exception If something goes wrong
     */
    @Override
    protected void onChange() throws Exception {

        this.s3Object = null;
        statSelf();
    }

    /**
     * Creates this file as a folder.
     *
     * @throws Exception If something goes wrong
     */
    @Override
    protected void doCreateFolder() throws Exception {

        this.statSelf();

        if (!isRoot()) {
            S3ClientWrapper client = this.fileSystem.getClient();
            try {
                // See https://www.codejava.net/aws/create-folder-examples
                String folder = (this.relPath.endsWith("/") ? this.relPath : this.relPath + '/');  // Folder indicator

                PutObjectRequest request = PutObjectRequest.builder() //
                        .bucket(getS3BucketName()) //
                        .key(folder) //
                        .build();
                client.s3Client.putObject(request, RequestBody.empty());
                return;

                // Remember that the S3 Management Console may need a refresh for new folders to show up
            } catch (Exception e) {
                this.fileSystem.putClient(client);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_CREATE_FOLDER_ERROR"));  // Recycled
    }

    /**
     * Gets the last modification date and time.
     *
     * @return The last modification time in milliseconds
     * @throws Exception If something goes wrong
     */
    @Override
    protected long doGetLastModifiedTime() throws Exception {

        this.statSelf();

        // May be null if an anticipated file is queried (e.g. abc.txt.sha)
        try {
            Instant modtime = this.s3Object.lastModified();
            if (modtime != null) return Date.from(modtime).getTime();
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_UNKNOWN_MOD_TIME_ERROR"));  // Recycled
    }

    /**
     * Sets the last modified time of this file (not supported by Amazon S3).
     *
     * @param modtime Last modification time in milliseconds
     * @return Successful?
     * @throws Exception If something goes wrong
     */
    @Override
    protected boolean doSetLastModifiedTime(long modtime) throws Exception {
        return false;  // Not supported
    }

    /**
     * Deletes the file.
     *
     * @throws Exception If something goes wrong
     */
    @Override
    protected void doDelete() throws Exception {

        this.statSelf();

        if (!isRoot()) {
            S3ClientWrapper client = this.fileSystem.getClient();
            try {
                String relpathfileorfolder = this.relPath;
                if (this.isFolder()) {
                    relpathfileorfolder = (this.relPath.endsWith("/") ? this.relPath : this.relPath + '/');  // Folder indicator
                }

                // Works also for folders
                DeleteObjectRequest request = DeleteObjectRequest.builder() //
                        .bucket(getS3BucketName()) //
                        .key(relpathfileorfolder) //
                        .build();
                client.s3Client.deleteObject(request);
                return;
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                this.fileSystem.putClient(client);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(GDriveClientWrapper.class.getName() + "_DELETE_ERROR")); // Recycled
    }

    /**
     * Rename the file.
     * <p>
     * Renaming is not currently supported by Amazon S3. The S3 Management Console does not support it
     * and neither does VFSLib.
     *
     * @throws Exception If something goes wrong
     */
    @Override
    protected void doRename(FileObject newfile) throws Exception {

        // Not supported
        super.doRename(newfile); // Throws

        // Feel free to experiment with the code below, don't forget to activate Capability.RENAME in S3FileProvider
        /*
        this.statSelf();

        if (!isRoot()) {
            S3ClientWrapper client = this.fileSystem.getClient();
            try {
                // See https://www.baeldung.com/java-amazon-s3-rename-files-folders
                String bucketname = getS3BucketName();

                if (this.isFile()) {

                    // Copy and delete source file, still no simple rename available in API v2
                    CopyObjectRequest copyrequest = CopyObjectRequest.builder() //
                            .sourceBucket(bucketname) //
                            .sourceKey(this.relPath) // Source
                            .destinationBucket(bucketname) //
                            .destinationKey(((S3FileObject) newfile).relPath) // Target
                            .build();
                    client.s3Client.copyObject(copyrequest);

                    DeleteObjectRequest deleterequest = DeleteObjectRequest.builder() //
                            .bucket(bucketname) //
                            .key(this.relPath) // Source
                            .build();
                    client.s3Client.deleteObject(deleterequest);
                    return;
                } else if (this.isFolder()) {
                    String relpathfolder = (this.relPath.endsWith("/") ? this.relPath : this.relPath + '/');  // Folder indicator
                    S3FileObject s3obj = (S3FileObject) newfile;
                    String relpathfoldernew = (s3obj.relPath.endsWith("/") ? s3obj.relPath : s3obj.relPath + '/');

                    // Copy and delete source children, still no simple rename available in API v2
                    ((S3FileObject) newfile).doCreateFolder();

                    ListObjectsV2Request listrequest = ListObjectsV2Request.builder() //
                            .bucket(bucketname) //
                            .prefix(relpathfolder) // Source
                            .delimiter("/")
                            .build();
                    ListObjectsV2Response listresponse = client.s3Client.listObjectsV2(listrequest);

                    // TODO Add recursion for subfolders via CommonPrefix's
                    List<S3Object> s3objects = listresponse.contents();
                    for (S3Object s3object : s3objects) {
                        String newkey = relpathfoldernew + s3object.key().substring(relpathfolder.length());

                        // Copy object to destination folder
                        CopyObjectRequest copyrequest = CopyObjectRequest.builder() //
                                .sourceBucket(bucketname) //
                                .sourceKey(s3object.key()) // Source
                                .destinationBucket(bucketname) //
                                .destinationKey(newkey) // Target
                                .build();
                        client.s3Client.copyObject(copyrequest);

                        // Delete object from source folder
                        DeleteObjectRequest deleterequest = DeleteObjectRequest.builder() //
                                .bucket(bucketname) //
                                .key(s3object.key()) // Source
                                .build();
                        client.s3Client.deleteObject(deleterequest);
                    }

                    DeleteObjectRequest deleterequest = DeleteObjectRequest.builder() //
                            .bucket(bucketname) //
                            .key(relpathfolder) // Source
                            .build();
                    client.s3Client.deleteObject(deleterequest);
                    return;
                }
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                this.fileSystem.putClient(client);
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_RENAME_ERROR"));  // Recycled
         */
    }

    /**
     * Lists the children of this file.
     *
     * @return The list of children, may be empty
     * @throws Exception If something goes wrong
     */
    @Override
    protected FileObject[] doListChildrenResolved() throws Exception {

        FileObject[] array = FileObject.EMPTY_ARRAY;
        S3ClientWrapper client = this.fileSystem.getClient();
        FileSystemManager fsmanager = getFileSystem().getFileSystemManager();

        statSelf();

        try {
            // See https://docs.aws.amazon.com/sdk-for-java/latest/developer-guide/examples-s3-objects.html#list-object

            // Must be compatible with folders created with AWS console.
            // Found here: http://stackoverflow.com/questions/1939743/amazon-s3-boto-how-to-create-folder

            String relpathfolder = (this.relPath.endsWith("/") ? this.relPath : this.relPath + '/');  // Folder indicator

            ListObjectsV2Request listrequest = ListObjectsV2Request.builder() //
                    .bucket(getS3BucketName()) //
                    .prefix(isRoot() ? "" : relpathfolder)
                    .delimiter("/")
                    .build();
            ListObjectsV2Response listresponse = client.s3Client.listObjectsV2(listrequest);

            // 1. step: Get folders (prefixes here)
            List<CommonPrefix> folders = listresponse.commonPrefixes();
            for (CommonPrefix nextfolder : folders) {
                if (nextfolder.prefix().equals(relpathfolder)) continue; // Self-reference

                String folder = nextfolder.prefix();  // testfolder/
                folder = Arrays.stream(folder.split("([/])")).reduce((first, second) -> second).orElse(null);  // Need only last part
                if (folder != null) {
                    FileObject fileobj = this.fileSystem.resolveFile(
                            fsmanager.resolveName(getName(), UriParser.encode(folder), NameScope.CHILD));
                    array = (FileObject[]) JavaUtils.addToArray(array, fileobj);

                    // Populate required fields, no need to call statSelf() again
                    S3FileObject s3obj = (S3FileObject) fileobj;
                    s3obj.s3Object = null;
                    s3obj.s3Folder = nextfolder;
                }
            }

            // 2. step: Get files (objects here)
            List<S3Object> children = listresponse.contents();
            for (S3Object child : children) {
                if (child.key().equals(relpathfolder)) continue; // Self-reference

                String file = child.key();  // testfolder/check2_16.png
                file = Arrays.stream(file.split("([/])")).reduce((first, second) -> second).orElse(null);  // Need only last part
                if (file != null) {
                    FileObject fileobj = this.fileSystem.resolveFile(
                            fsmanager.resolveName(getName(), UriParser.encode(file), NameScope.CHILD));
                    array = (FileObject[]) JavaUtils.addToArray(array, fileobj);

                    // Populate required fields, no need to call statSelf() again
                    S3FileObject s3obj = (S3FileObject) fileobj;
                    s3obj.s3Object = child;
                    s3obj.s3Folder = null;
                }
            }
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
            throw e;
        } finally {
            this.fileSystem.putClient(client);
        }
        return array;
    }

    /**
     * Lists the children of this file.
     *
     * @return The list of children, may be empty
     * @throws Exception If something goes wrong
     */
    @Override
    protected String[] doListChildren() throws Exception {

        FileObject[] children = this.doListChildrenResolved();
        return Arrays.stream(children).map(String::valueOf).toArray(String[]::new);
    }

    /**
     * Returns the size of the file content (in bytes).
     *
     * @return The size, -1 if not available
     * @throws Exception If something goes wrong
     */
    @Override
    protected long doGetContentSize() throws Exception {

        statSelf();

        // May be null if an anticipated file is queried (e.g. abc.txt.sha)
        try {
            return this.s3Object.size();
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_UNKNOWN_SIZE_ERROR"));  // Recycled
    }

    /**
     * Provides the instance to model random access for files.
     *
     * @return The instance
     * @throws Exception If something goes wrong
     */
    @Override
    protected RandomAccessContent doGetRandomAccessContent(RandomAccessMode mode) throws Exception {
        return new VFSRandomAccessContent(this, mode);
    }

    /**
     * Creates an input stream to read the file content from.
     *
     * @return The input stream
     * @throws Exception If something goes wrong
     */
    @Override
    protected InputStream doGetInputStream() throws Exception {

        this.statSelf();

        if (!FileType.FILE.equals(getType())) { // Required to exist
            String text = VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_READ_NOT_FILE_ERROR");  // Recycled
            throw new FileSystemException(text, getName());
        }

        S3ClientWrapper client = this.fileSystem.getClient();

        // VFS-113: Avoid NPE
        synchronized (this.fileSystem) {
            try {
                GetObjectRequest request = GetObjectRequest.builder() //
                        .key(s3Object.key()) //
                        .bucket(getS3BucketName()) //
                        .build();
                ResponseBytes<GetObjectResponse> objectBytes = client.s3Client.getObjectAsBytes(request);
                InputStream inputstream = objectBytes.asInputStream();

                return new S3InputStream(client, inputstream);
            } finally {
                // For client release and stream close see S3InputStream
            }
        }
    }

    /**
     * Creates an output stream to write the file content to.
     *
     * @param append Not supported currently, always created freshly
     * @return The output stream
     * @throws Exception If something goes wrong
     */
    @Override
    protected OutputStream doGetOutputStream(boolean append) throws Exception {

        this.statSelf();

        if (!FileType.FILE.equals(getType()) && !FileType.IMAGINARY.equals(getType())) {
            String text = VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_WRITE_NOT_FILE_ERROR");  // Recycled
            throw new FileSystemException(text, getName());
        }

        final S3ClientWrapper client = this.fileSystem.getClient();

        // VFS-113: Avoid NPE
        synchronized (this.fileSystem) {
            try {
                // See https://www.baeldung.com/java-aws-s3

                PipedOutputStream postream = new PipedOutputStream();
                PipedInputStream pistream = new PipedInputStream(postream, 4096);  // See FileSystemCopier class

                // Amazon S3 Javadoc:
                // "When uploading directly from an input stream, content length must be specified before data
                // can be uploaded to Amazon S3. If not provided, the library will have to buffer the contents
                // of the input stream in order to calculate it. Amazon S3 explicitly requires that the content
                // length be sent in the request headers before any of the data is sent."

                // Means: If not provided the whole data from the stream is cached to get the content length (JavaHeapSpace error)!

                // Content length has been set as attribute for this file?
                Long len = null;
                try {
                    // Throws FileSystemException
                    len = Long.parseLong(String.valueOf(super.getContent().getAttribute(VFSLibConstants.ATTR_CONTENT_LENGTH)));
                } catch (Exception e) {
                    VFSLibSettings.log(Level.WARNING, e);
                }
                Long finallen = len;

                PutObjectRequest request = PutObjectRequest.builder() //
                        .bucket(getS3BucketName()) //
                        .key(this.relPath) //
                        .build();
                Thread thread = new Thread(() -> {
                    try {
                        client.s3Client.putObject(request, RequestBody.fromInputStream(pistream, finallen)); // Throws
                    } catch (Exception e) {
                        VFSLibSettings.log(Level.WARNING, e);
                    }
                }, VFSLibSettings.LOG_PREFIX + "Put Object Requester");
                thread.setPriority(Thread.MIN_PRIORITY);
                thread.start();

                return new S3OutputStream(client, postream, thread);  // Release client afterwards
            } catch (Exception e) {
                VFSLibSettings.log(Level.WARNING, e);
            } finally {
                // For client release and stream close see S3OutputStream
            }
        }
        throw new FileSystemException(VFSLibSettings.getUserText(DbxClientWrapper.class.getName() + "_OUTPUTSTREAM_ERROR"));  // Recycled
    }

    /**
     * An <code>InputStream</code> that monitors for end-of-file.
     */
    protected class S3InputStream extends MonitorInputStream {

        protected S3ClientWrapper client;

        public S3InputStream(S3ClientWrapper client, InputStream in) {

            super(in);
            this.client = client;
        }

        @Override
        protected void onClose() {

            // No need to close stream here
            fileSystem.putClient(this.client);
        }
    }

    /**
     * An <code>OutputStream</code> that wraps a Amazon S3 <code>OutputStream</code>.
     */
    protected class S3OutputStream extends MonitorOutputStream {

        protected S3ClientWrapper client;
        protected Thread requester;

        public S3OutputStream(S3ClientWrapper client, OutputStream out, Thread requester) {

            super(out);
            this.client = client;
            this.requester = requester;
        }

        @Override
        protected void onClose() {

            try {
                // Wait for request thread to be finished, size does not fit otherwise etc.
                this.requester.join();
            } catch (Exception ignored) {
            }

            // No need to close stream here
            fileSystem.putClient(this.client);
        }
    }

    /**
     * Provides the VFSLib instance for this filesystem.
     *
     * @return The instance
     */
    public VFSLib getVFSLib() {
        return this.fileSystem.getVFSLib();
    }

    /**
     * Provides the <code>ClientWrapper</code> implementation for this file system.
     *
     * @return The client wrapper holding the low-level client
     * @throws IOException If an I/O error occurs
     * @since 2.8
     */
    public S3ClientWrapper getS3ClientWrapper() throws IOException {
        return this.fileSystem.getClient();
    }

    /**
     * Returns the attributes specific for cloud file systems (e.g. "Content-Length" for proper upload)
     *
     * @return The map
     */
    @Override
    protected Map<String, Object> doGetAttributes() {

        synchronized (this) {
            if (this.attributes == null) this.attributes = new HashMap<>(0);
        }
        return this.attributes;
    }

    /**
     * Sets an attribute of this cloud file.
     *
     * @param attrName The name (ATTR_CONTENT_LENGTH|...)
     * @param value    The value
     */
    @Override
    protected void doSetAttribute(String attrName, Object value) {

        doGetAttributes();  // Make sure cache exists
        this.attributes.put(attrName, value);
    }
}
