/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.lang;

import com.lf.vfslib.core.VFSLibSettings;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.logging.Level;


/**
 * This class holds various methods for common use with reflection.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class ReflectionUtils {


    /**
     * Constructor method for i18n purposes only.
     *
     * @throws InstantiationException Error indication
     */
    public ReflectionUtils() throws InstantiationException {

        if (!java.beans.Beans.isDesignTime()) {
            throw new InstantiationException("Do not use this constructor!");
        }
    }

    /**
     * Extracts the value of the instance variable via reflection (also protected, private etc.).
     * <p>
     * The <code>clazz</code> parameter may be one of the subclasses of <code>target</code>.
     *
     * @param target The object to be accessed
     * @param name   The name of the variable to access
     * @param clazz  The (sub)class of interest, target class if <code>null</code>
     * @return Instance variable value, may be <code>null</code>
     * @throws NullPointerException   If the parameters are <code>null</code>
     * @throws NoSuchFieldException   Error indication
     * @throws IllegalAccessException Error indication
     */
    public static Object getDeclaredField(Object target, Class clazz, String name) throws NoSuchFieldException,
            IllegalAccessException {

        if (clazz == null) clazz = target.getClass();

        Field field = clazz.getDeclaredField(name);
        field.setAccessible(true);  // Override access limitations for private, protected etc.
        return field.get(target);
    }

    /**
     * Extracts the value of the instance variable via reflection (also protected, private etc.).
     *
     * @param target The object to be accessed
     * @param name   The name of the variable to access
     * @return Instance variable value, may be <code>null</code>
     * @throws NullPointerException   If the parameters are <code>null</code>
     * @throws NoSuchFieldException   Error indication
     * @throws IllegalAccessException Error indication
     */
    public static Object getDeclaredField(Object target, String name) throws NoSuchFieldException,
            IllegalAccessException {

        Field field = target.getClass().getDeclaredField(name);
        field.setAccessible(true);  // Override access limitations for private, protected etc.
        return field.get(target);
    }

    /**
     * Creates a new instance of the given class using reflection (also used for the paramtypes classes).
     *
     * @param classname  The object to be accessed
     * @param paramtypes The parameter types for constructor, <code>null</code> = default constructor
     * @param initargs   The arguments for constructor, <code>null</code> = default constructor
     * @return Instance reference, <code>null</code> if an error occurred
     */
    public static Object newInstance(String classname, String[] paramtypes, Object... initargs) {

        try {
            if (paramtypes == null) {
                // Default parameterless constructor must exist here
                return Class.forName(classname).newInstance();
            } else {
                // http://stackoverflow.com/questions/6241513/how-to-create-instance-of-a-class-with-the-parameters-in-the-constructor-using-r
                Class[] classes = new Class[0];
                for (String next : paramtypes) {
                    classes = (Class[]) JavaUtils.addToArray(classes, Class.forName(next));
                }
                return Class.forName(classname).getConstructor(classes).newInstance(initargs);
            }
        } catch (Throwable e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        return null;
    }

    /**
     * Invokes a method using reflection.
     *
     * @param target     The object to be accessed
     * @param methodname The method to be invoked
     * @param paramtypes The parameter types for method call, <code>null</code> = parameterless method
     * @param args       The arguments for method call
     * @return Method return valiu, <code>null</code> otherwise (also if errors did occur)
     */
    public static Object invoke(Object target, String methodname, String[] paramtypes, Object... args) {

        try {
            if (paramtypes == null) {
                Method method = null;
                try {
                    method = target.getClass().getMethod(methodname);
                }  // Parameterless
                catch (Throwable ignored) {
                }  // Thrown for private methods

                if (method == null) method = target.getClass().getDeclaredMethod(methodname);  // Parameterless

                method.setAccessible(true);
                return method.invoke(target);
            } else {
                Method method = null;
                Class[] classes = new Class[0];
                for (String next : paramtypes) {
                    classes = (Class[]) JavaUtils.addToArray(classes, Class.forName(next));
                }
                try {
                    method = target.getClass().getMethod(methodname, classes);
                } catch (Throwable ignored) {
                }  // Thrown for private methods

                if (method == null) method = target.getClass().getMethod(methodname, classes);
                method.setAccessible(true);
                return method.invoke(target, args);
            }
        } catch (Throwable e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        return null;
    }

    /**
     * Invokes a static (class) method using reflection.
     *
     * @param classname  The class to be accessed
     * @param methodname The method to be invoked
     * @param paramtypes The parameter types for method call, <code>null</code> = parameterless method
     * @param args       The arguments for method call
     * @return Method return value, <code>null</code> otherwise (also if errors did occur)
     * @since 2.8
     */
    public static Object invokeStatic(String classname, String methodname, String[] paramtypes, Object... args) {

        try {
            Class clazz = Class.forName(classname);
            if (paramtypes == null) {
                Method method = null;
                try {
                    method = clazz.getMethod(methodname);
                }  // Parameterless
                catch (Exception ignored) {
                }  // Thrown for private methods

                if (method == null) method = clazz.getDeclaredMethod(methodname);  // Parameterless

                method.setAccessible(true);
                return method.invoke(null);  // static
            } else {
                Method method = null;
                Class[] classes = new Class[0];
                for (String next : paramtypes) {
                    classes = (Class[]) JavaUtils.addToArray(classes, Class.forName(next));
                }
                try {
                    method = clazz.getMethod(methodname, classes);
                } catch (Exception ignored) {
                }  // Thrown for private methods

                if (method == null) method = clazz.getDeclaredMethod(methodname, classes);

                method.setAccessible(true);
                return method.invoke(null, args);  // static
            }
        } catch (Throwable e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        return null;
    }

    /**
     * Extracts the value of the static field via reflection (only public).
     *
     * @param classname The class name to be accessed
     * @param fieldname The name of the field to access
     * @return Static field value, may be <code>null</code>
     * @throws NullPointerException   If the parameters are <code>null</code>
     * @throws ClassNotFoundException Error indication
     * @throws NoSuchFieldException   Error indication
     * @throws IllegalAccessException Error indication
     * @since 2.8
     */
    public static Object getStaticField(String classname, String fieldname) {

        try {
            Field field = Class.forName(classname).getField(fieldname);
            field.setAccessible(true);  // Override access limitations for private, protected etc.
            return field.get(null);
        } catch (Throwable e) {
            VFSLibSettings.log(Level.WARNING, e);
        }
        return null;
    }
}
