/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.io;

import com.lf.vfslib.core.VFSLibSettings;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.provider.AbstractFileObject;
import org.apache.commons.vfs2.provider.AbstractRandomAccessStreamContent;
import org.apache.commons.vfs2.util.RandomAccessMode;

import java.io.DataInputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;


/**
 * Random access content.
 * <p>
 * For example the Dropbox API does not provide native random access, so this implementation is not optimized.
 *
 * @author Axel Schwolow
 * @created 2024-03-14
 * @since 2.8
 */
public class VFSRandomAccessContent extends AbstractRandomAccessStreamContent {


    /**
     * Position in file.
     */
    protected long filePointer = 0;
    /**
     * The underlying file object.
     */
    protected AbstractFileObject fileObject;
    /**
     * The data input stream.
     */
    protected DataInputStream dataInputStream = null;
    /**
     * The original input stream from the provider.
     */
    protected InputStream inputStream = null;


    /**
     * Constructor method.
     *
     * @param fileobject The underlying file object
     * @param mode       The mode for random access
     * @throws NullPointerException If parameters are <code>null</code>
     * @since 1.6
     */
    public VFSRandomAccessContent(AbstractFileObject fileobject, RandomAccessMode mode) {

        super(mode);

        if (fileobject == null) throw new NullPointerException();

        this.fileObject = fileobject;
    }

    /**
     * Provides the current position in file.
     *
     * @return The position
     * @since 1.6
     */
    @Override
    public long getFilePointer() {
        return this.filePointer;
    }

    /**
     * Seeks the given position in file.
     *
     * @param pos The position
     * @throws IOException If something goes wrong
     * @since 1.6
     */
    @Override
    public void seek(long pos) throws IOException {

        if (pos == this.filePointer) return;  // No change
        else if (pos < 0) {
            String text = VFSLibSettings.getUserText(VFSRandomAccessContent.class.getName() + "_RANDOM_ACCESS_INVALID_POSITION_ERROR");
            throw new FileSystemException(text, new Object[]{pos});
        }
        if (this.dataInputStream != null) this.close();  // Finish previous seek
        this.filePointer = pos;
    }

    /**
     * Creates an input stream at the current position.
     *
     * @throws IOException If something goes wrong
     * @since 1.6
     */
    @Override
    protected DataInputStream getDataInputStream() throws IOException {

        if (this.dataInputStream != null) return this.dataInputStream;

        this.inputStream = this.fileObject.getInputStream();

        // For example the Dropbox input stream does not support random access natively. Overread bytes before position.
        for (int i = 0; i < this.filePointer; i++) {
            this.inputStream.read();
        }

        this.dataInputStream = new DataInputStream(new FilterInputStream(this.inputStream) {

            @Override
            public int read() throws IOException {
                int ret = super.read();
                if (ret > -1) filePointer++;
                return ret;
            }

            @Override
            public int read(byte[] b) throws IOException {
                int ret = super.read(b);
                if (ret > -1) filePointer += ret;
                return ret;
            }

            @Override
            public int read(byte[] b, int off, int len) throws IOException {
                int ret = super.read(b, off, len);
                if (ret > -1) filePointer += ret;
                return ret;
            }

            @Override
            public void close() throws IOException {
                VFSRandomAccessContent.this.close();
            }
        });
        return this.dataInputStream;
    }

    /**
     * Closes this random access content.
     *
     * @throws IOException If something goes wrong
     * @since 1.6
     */
    @Override
    public void close() throws IOException {

        if (this.dataInputStream != null) {
            this.inputStream.close();

            // This is to avoid recursive close
            DataInputStream oldstream = this.dataInputStream;
            this.dataInputStream = null;
            oldstream.close();
            this.inputStream = null;
        }
    }

    /**
     * Provides the total length of the underlying file.
     *
     * @return The length
     * @throws IOException If something goes wrong
     * @since 1.6
     */
    @Override
    public long length() throws IOException {
        return this.fileObject.getContent().getSize();
    }
}
