/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.gdrive;

import com.google.api.services.drive.Drive;
import com.google.api.services.drive.model.File;
import com.lf.vfslib.VFSLib;
import com.lf.vfslib.core.VFSLibSettings;
import com.lf.vfslib.net.ClientPool;
import com.lf.vfslib.net.ClientWrapper;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.provider.GenericFileName;

import java.util.List;
import java.util.logging.Level;


/**
 * Wraps a Google Drive client.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class GDriveClientWrapper implements ClientWrapper {


    /**
     * The root file name.
     */
    protected GenericFileName root = null;
    /**
     * The file system options.
     */
    protected FileSystemOptions fileSystemOptions = null;
    /**
     * The parental object.
     */
    protected VFSLib vfsLib = null;
    /**
     * The file provider.
     */
    protected GDriveFileProvider provider = null;
    /**
     * The low-level client.
     */
    protected Drive driveClient = null;


    /**
     * Constructor method (do not use).
     *
     * @throws InstantiationException Error indication
     */
    public GDriveClientWrapper() throws InstantiationException {

        if (!java.beans.Beans.isDesignTime()) {
            throw new InstantiationException("Do not use this constructor!");
        }
    }

    /**
     * Constructor method.
     *
     * @param root     The file system root
     * @param options  The file system specific configuration
     * @param vfslib   The parental object, shared instance if <code>null</code>
     * @param provider The file provider
     * @throws org.apache.commons.vfs2.FileSystemException If something goes wrong
     */
    GDriveClientWrapper(GenericFileName root, FileSystemOptions options, VFSLib vfslib, GDriveFileProvider provider)
            throws FileSystemException {

        this.root = root;
        this.fileSystemOptions = options;
        this.vfsLib = vfslib != null ? vfslib : VFSLib.getSharedInstance();
        this.provider = provider;

        createClient();  // Fail fast
    }

    /**
     * Establish connection to Google Drive based on the configured options.
     *
     * @throws org.apache.commons.vfs2.FileSystemException If something goes wrong
     */
    protected void createClient() throws FileSystemException {
        try {
            // Create client first
            this.driveClient = GDriveClientFactory.createClient(this.fileSystemOptions, this.provider.getApplicationName(),
                    this.provider.getAuthFlow());

            // Fail fast like SFTP: access root files, provoke exceptions
            Drive.Files.List request = this.driveClient.files().list();
            request.setPageSize(1);
            request.setPageToken(null);
            request.setFields("files(id,name)");
            List<File> files = request.execute().getFiles();
            files.forEach(f -> VFSLibSettings.log(Level.CONFIG, f.getName()));

        } catch (Exception e) {
            String text = VFSLibSettings.getUserText(GDriveClientWrapper.class + "_CONNECT_ERROR");
            throw new FileSystemException(text.replaceAll("(%server%)", String.valueOf(this.root)), this.root, e);
        }
    }

    /**
     * Provides access to the internal Google Drive client.
     *
     * @return The client reference
     */
    public Drive getClient() {
        return this.driveClient;
    }

    /**
     * Returns the options for this file system.
     *
     * @return The options
     */
    public FileSystemOptions getFileSystemOptions() {
        return this.fileSystemOptions;
    }

    /**
     * Let's the underlying client disconnect from the server.
     *
     * @return Successful?
     */
    // Implements ClientWrapper interface
    public boolean disconnectClient() {
        return true;  // Nothing to do here
    }

    /**
     * Provides the unique URL identifier used to manage this client instance.
     * <p>
     * Something like "gdrive://user@drive.google.com".
     *
     * @return The URL
     */
    // Implements ClientWrapper interface
    public String getURLID() {
        return this.toString();
    }

    /**
     * Gets an idle client from the pool or creates a fresh new instance.
     *
     * @param root     The root path
     * @param options  The file system options
     * @param vfslib   The parental object, shared instance if <code>null</code>
     * @param provider The file provider
     * @return Wrapper instance
     * @throws org.apache.commons.vfs2.FileSystemException If an I/O error occurs
     */
    public static GDriveClientWrapper getGDriveClientWrapper(GenericFileName root, FileSystemOptions options,
                                                             VFSLib vfslib, GDriveFileProvider provider) throws FileSystemException {

        if (vfslib == null) vfslib = VFSLib.getSharedInstance();

        // Get connection from pool if possible
        ClientPool pool = ClientPool.getSharedInstance();
        try {
            // gdrive://user@drive.google.com
            ClientWrapper recycled = pool.getFreeClient(getClientURLID(root, options, vfslib));
            if (recycled != null) return (GDriveClientWrapper) recycled;
        } catch (Exception e) {
            VFSLibSettings.log(Level.WARNING, e);
        }

        // Create a fresh new instance
        GDriveClientWrapper wrapper = new GDriveClientWrapper(root, options, vfslib, provider);
        pool.addClient(wrapper);  // Locked automatically
        return wrapper;
    }

    /**
     * Releases the given client from the pool.
     *
     * @param wrapper The wrapper
     */
    public static void unlockGDriveClientWrapper(GDriveClientWrapper wrapper) {
        ClientPool.getSharedInstance().unlockClient(wrapper);
    }

    /**
     * Provides a textual representation.
     *
     * @return The text
     */
    @Override
    public String toString() {
        return getClientURLID(this.root, this.fileSystemOptions, this.vfsLib);
    }

    /**
     * Provides a textual representation for this connection.
     *
     * @param root    The root path
     * @param options The file system options
     * @param vfslib  The parental object, shared instance if <code>null</code>
     * @return The text
     */
    public static String getClientURLID(GenericFileName root, FileSystemOptions options, VFSLib vfslib) {

        if (vfslib == null) vfslib = VFSLib.getSharedInstance();

        GDriveFileSystemConfigBuilder builder = GDriveFileSystemConfigBuilder.getSharedInstance();
        String username = builder.getAccountDisplayName(options);
        if (username == null) username = "anonymous";

        return vfslib.getSchemeGoogleDrive() + "://" + username + '@' + root.getHostName();  // No port
    }
}
