/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.gdrive;

import com.google.api.client.googleapis.auth.oauth2.GoogleAuthorizationCodeFlow;
import com.google.api.client.googleapis.auth.oauth2.GoogleCredential;
import com.google.api.services.drive.Drive;
import org.apache.commons.vfs2.FileSystemOptions;

import java.io.IOException;
import java.security.GeneralSecurityException;


/**
 * Factory for Google Drive clients.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class GDriveClientFactory {


    /**
     * Constructor method (do not use).
     *
     * @throws InstantiationException Error indication
     */
    public GDriveClientFactory() throws InstantiationException {

        if (!java.beans.Beans.isDesignTime()) {
            throw new InstantiationException("Do not use this constructor!");
        }
    }

    /**
     * Creates a new connection to the server.
     *
     * @param options  The file system options
     * @param appname  The application name (required)
     * @param authflow The request configuration
     * @return The new client
     */
    public static Drive createClient(FileSystemOptions options, String appname, GoogleAuthorizationCodeFlow authflow)
            throws GeneralSecurityException, IOException {

        // See Java example https://developers.google.com/drive/api/quickstart/java?hl=de

        // All required data is provided by the file system options
        GDriveFileSystemConfigBuilder builder = GDriveFileSystemConfigBuilder.getSharedInstance();
        String accesstoken = builder.getAccessToken(options);
        Integer connecttimeoutseconds = builder.getConnectTimeoutSeconds(options);
        Integer readtimeoutseconds = builder.getReadTimeoutSeconds(options);

        GoogleCredential credential = new GoogleCredential().setAccessToken(accesstoken);

        Drive.Builder drivebuilder = new Drive.Builder(authflow.getTransport(), authflow.getJsonFactory(), credential);

        // Avoid timeouts while downloading large files
        // See https://stackoverflow.com/questions/13026613/how-to-avoid-read-timeout-when-requesting-all-files-google-drive-api?rq=4
        if (connecttimeoutseconds != null || readtimeoutseconds != null) {
            drivebuilder.setHttpRequestInitializer(httpRequest -> {
                credential.initialize(httpRequest);
                if (connecttimeoutseconds != null) {
                    httpRequest.setConnectTimeout(connecttimeoutseconds * 1000); // Default: 20 seconds
                }
                if (readtimeoutseconds != null) {
                    httpRequest.setReadTimeout(readtimeoutseconds * 1000); // Default: 20 seconds
                }
            });
        }
        if (appname != null) drivebuilder.setApplicationName(appname);  // Logger warns if not set
        return drivebuilder.build();
    }
}
