/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.vfslib.dropbox;

import com.dropbox.core.DbxAppInfo;
import com.dropbox.core.DbxRequestConfig;
import com.lf.vfslib.VFSLib;
import org.apache.commons.vfs2.*;
import org.apache.commons.vfs2.provider.AbstractOriginatingFileProvider;
import org.apache.commons.vfs2.provider.GenericFileName;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;


/**
 * A provider to access the files hosted by Dropbox.
 * <p>
 * This class coordinates the whole Dropbox functionality used by VFS2.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.6
 */
public class DbxFileProvider extends AbstractOriginatingFileProvider {


    /**
     * The Dropbox application info.
     */
    protected DbxAppInfo appInfo = null;
    /**
     * The Dropbox request configuration.
     */
    protected DbxRequestConfig reqConfig = null;
    /**
     * The parental object.
     */
    protected VFSLib vfsLib = null;


    /**
     * The provider's capabilities.
     */
    static final Collection<Capability> capabilities =
            Collections.unmodifiableCollection(Arrays.asList(
                    //Capability.APPEND_CONTENT,
                    //Capability.ATTRIBUTES,
                    //Capability.COMPRESS,
                    Capability.CREATE,
                    Capability.DELETE,
                    //Capability.DIRECTORY_READ_CONTENT,
                    //Capability.DISPATCHER,
                    //Capability.FS_ATTRIBUTES,
                    Capability.GET_LAST_MODIFIED, // Currently not supported for folders
                    Capability.GET_TYPE,
                    //Capability.JUNCTIONS,
                    //Capability.LAST_MODIFIED,
                    Capability.LIST_CHILDREN,
                    //Capability.MANIFEST_ATTRIBUTES,
                    Capability.RANDOM_ACCESS_READ,
                    //Capability.RANDOM_ACCESS_WRITE,
                    Capability.READ_CONTENT,
                    Capability.RENAME,
                    //Capability.SET_LAST_MODIFIED_FILE,
                    //Capability.SET_LAST_MODIFIED_FOLDER,
                    //Capability.SIGNING,
                    Capability.URI,
                    //Capability.VIRTUAL,
                    Capability.WRITE_CONTENT));


    /**
     * Constructor method.
     */
    public DbxFileProvider() {
    }

    /**
     * Constructor method.
     *
     * @param appinfo   The Dropbox application info (type is <code>com.dropbox.core.DbxAppInfo</code>)
     * @param reqconfig The Dropbox request configuration (type is <code>com.dropbox.core.DbxRequestConfig</code>)
     * @param vfslib    The parental object
     * @throws NullPointerException     If parameters are <code>null</code>
     * @throws IllegalArgumentException If a parameter is not of the expected type
     */
    public DbxFileProvider(Object appinfo, Object reqconfig, VFSLib vfslib) {

        super();

        if (appinfo == null) throw new NullPointerException();
        if (!(appinfo instanceof DbxAppInfo)) {
            throw new IllegalArgumentException("Parameter appinfo must be of type DbxAppInfo");
        }
        if (reqconfig == null) throw new NullPointerException();
        if (!(reqconfig instanceof DbxRequestConfig)) {
            throw new IllegalArgumentException("Parameter reqconfig must be of type DbxRequestConfig");
        }
        if (vfslib == null) throw new NullPointerException();

        // Using this info an authorize URL can be acquired from Dropbox
        // in order to let the user allow this application to access his Dropbox resources.
        this.appInfo = (DbxAppInfo) appinfo;
        this.reqConfig = (DbxRequestConfig) reqconfig;
        this.vfsLib = vfslib;

        setFileNameParser(DbxFileNameParser.getSharedInstance());
    }

    /**
     * Provides the provider's Dropbox application info.
     *
     * @return The app info
     */
    public DbxAppInfo getAppInfo() {
        return this.appInfo;
    }

    /**
     * Provides the provider's Dropbox request configuration.
     *
     * @return The app info
     */
    public DbxRequestConfig getRequestConfig() {
        return this.reqConfig;
    }

    /**
     * Creates a Dropbox file system.
     *
     * @param name    The file name
     * @param options The file system options
     * @throws FileSystemException If something goes wrong
     */
    @Override
    protected FileSystem doCreateFileSystem(FileName name, FileSystemOptions options) throws FileSystemException {

        // Create the file system
        GenericFileName rootName = (GenericFileName) name;

        // Check out for a free connection here to detect errors
        DbxClientWrapper client = DbxClientWrapper.getDbxClientWrapper(rootName, options, this.vfsLib, this);
        DbxClientWrapper.unlockDbxClientWrapper(client);

        return new DbxFileSystem(rootName, options, this.vfsLib, this);
    }

    /**
     * Provides the builder instance responsible for proper file system options.
     *
     * @return The builder
     */
    @Override
    public FileSystemConfigBuilder getConfigBuilder() {
        return DbxFileSystemConfigBuilder.getSharedInstance();
    }

    /**
     * Provides the provider's capabilities.
     *
     * @return The collection
     */
    public Collection<Capability> getCapabilities() {
        return DbxFileProvider.capabilities;
    }
}
