/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.javainfo.gui;

import java.applet.Applet;
import java.awt.*;
import java.io.*;
import java.net.URL;
import java.text.Collator;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.*;


/**
 * Small GUI/CLI application to let the user send Java system properties to a servlet.
 * <p/>
 * This class was inspired by <a href="http://lopica.sourceforge.net/os.html">...</a> providing a list of possible
 * values for the <code>os.name</code>, <code>os.version</code>, and <code>os.arch</code>
 * properties found on different platforms. This information can be gathered by a web database
 * and may be a basis for values used with Web Start applications where platform specific
 * native libs are needed.
 * <p/>
 * By default this application opens a simple dialog with buttons to send data to the
 * web servlet or to save the data into a file. If started with the "-headless" parameter
 * then the output is printed on the command line (stdout). This is necessary for VMs
 * like those on iSeries machines (aka AS/400) which lack GUI support. Most operating
 * systems provide a way to save this output into a file which can then be copied to a
 * special web form.
 * <p/>
 * In order to prevent web robots from sending arbitrary content to the servlet, the XML
 * file contains a checksum generated from certain parts of the XML data. This way we can
 * make sure that only data portions generated by this executable is sent to the servlet.
 * <p/>
 * The current JNLP specification doesn't list the values for the os or arch attributes.
 * Instead it states that all values are valid as long as they match the values returned
 * by the system properties <code>os.name</code> or <code>os.arch</code>. Values that only
 * match the beginning of <code>os.name</code> or <code>os.arch</code> are also valid
 * (e.g. if you specify Windows and the <code>os.name</code> is Windows 98, for example,
 * everything works out fine).
 * <p/>
 * Remember that this class must be compatible with Java 1.4 (!) in order to support even
 * older Java VMs and is therefore very limited in GUI design and the set of Java features.
 * For this reason this class includes several methods from other classes of this package
 * since it must be runnable standalone.
 * <p/>
 * The API documentation of Java 1.1.8 lists the following properties to be available:
 * <p/>
 * <pre>
 *     java.version         Java version number
 *     java.vendor          Java vendor-specific string
 *     java.vendor.url      Java vendor URL
 *     java.home            Java installation directory
 *     java.class.version   Java class format version number
 *     java.class.path      Java class path
 *     os.name              Operating system name
 *     os.arch              Operating system architecture
 *     os.version           Operating system version
 *     file.separator       File separator ("/" on UNIX)
 *     path.separator       Path separator (":" on UNIX)
 *     line.separator       Line separator ("\n" on UNIX)
 *     user.name            User's account name
 *     user.home            User's home directory
 *     user.dir             User's current working directory
 * </pre>
 *
 * @author Axel Schwolow
 * @since 1.6
 */
public class JavaInfo extends Applet {


    /**
     * The icon to be displayed by the window.
     */
    protected final static String ICONPATH = "/com/lf/javainfo/resource/cup_32.png";
    /**
     * The deploy properties file.
     */
    protected final static String PROPSPATH = "/com/lf/javainfo/resource/deploy.properties";
    /**
     * The default name.
     */
    protected static String JAVAINFO_APP_NAME = "JavaInfo";
    /**
     * The URL to upload Java properties.
     */
    public static String SERVLET_URL = "n/a";
    /**
     * The URL of the Privacy Policy.
     */
    public static String POLICY_URL = "n/a";

    /**
     * Constructor method for i18n purposes only.
     * <p/>
     * Can only be used if beans design time is set (e.g. by external tools like the
     * <code>I18NExtractor</code>).
     *
     * @throws InstantiationException Error indication
     */
    public JavaInfo() throws InstantiationException {

        if (!java.beans.Beans.isDesignTime()) {
            throw new InstantiationException("Do not use this constructor!");
        }
    }

    /**
     * Implements the headless mode of this application.
     */
    private static void processCLI() {

        try {
            Properties props = new Properties();  // Avoid using JavaInfoSettings (Java 1.4)!
            try {
                props.load(JavaInfo.class.getResourceAsStream(PROPSPATH));
                if (System.getProperty("SERVLET_URL") != null) {
                    SERVLET_URL = System.getProperty("SERVLET_URL");
                } else {
                    SERVLET_URL = props.getProperty("SERVLET_URL");
                }
                POLICY_URL = props.getProperty("POLICY_URL");
            } catch (Exception ignored) {
            }

            // Print the system properties first
            byte[] data = JavaInfoUtils.generateData(JAVAINFO_APP_NAME);
            System.out.write(data);

            // Main menu with button functionality
            System.out.println("\n1 - " + getLocalizedText("JavaInfo.submit"));
            System.out.println("2 - " + getLocalizedText("JavaInfo.saveas"));
            System.out.println("3 - " + getLocalizedText("JavaInfo.exit") + '\n');

            String input = JavaInfo.getInput(getLocalizedText("JavaInfo.choose") + ' ', new String[]{"1", "2", "3"});
            if (input.equals("1")) {

                // Try to submit the properties to web service
                String address = SERVLET_URL + "?locale=" + JavaInfoUtils.LF_LOCALE;
                byte[] result = JavaInfoUtils.callWebServiceByPOST(new URL(address), data);

                // Status OK? If not, the service possibly answered with an error message.
                String status = JavaInfoUtils.checkResponse(result);
                if (status != null && status.equals("OK")) {
                    System.out.println('\n' + getLocalizedText("JavaInfo.ok_submit") + '\n');
                } else {
                    if (status != null) System.out.println('\n' + status + '\n');  // Print directly
                    else {
                        System.out.println('\n' + getLocalizedText("JavaInfo.error_submit") + '\n');
                    }
                }
            } else if (input.equals("2")) {

                // Save data to file. Ask for file name and check if the file already exists.
                System.out.println();
                String filename = null;
                while (filename == null) {
                    filename = JavaInfo.getInput(getLocalizedText("JavaInfo.filename") + ' ',
                            new String[0]);
                    if (filename.trim().isEmpty()) filename = null;
                }

                File target = new File(filename);
                if (!target.isAbsolute()) {

                    // Expand relative file name like 'props.xml' with working directory
                    String workdir = System.getProperties().getProperty("user.dir");
                    target = new File(workdir + File.separator + filename);
                }
                if (target.exists()) {
                    String text = getLocalizedText("JavaInfo.askoverwrite");
                    text = new String(JavaInfoUtils.replace(text.toCharArray(), "%FILE%", target.toString()));
                    input = JavaInfo.getInput('\n' + text + ' ',
                            new String[]{getLocalizedText("JavaInfo.askyes"),
                                    getLocalizedText("JavaInfo.askno")});
                    if (input.toLowerCase().equals(getLocalizedText("JavaInfo.askno"))) {
                        System.out.println('\n' + getLocalizedText("JavaInfo.notoverwritten"));
                        return;  // Exit
                    }
                }

                // Ok, let's store the data into the specified file
                try {
                    FileOutputStream fstream = new FileOutputStream(target);
                    fstream.write(data);
                    fstream.flush();
                    fstream.close();
                    System.out.println('\n' + getLocalizedText("JavaInfo.ok_write") + '\n');
                } catch (Exception e) {
                    System.out.println('\n' + getLocalizedText("JavaInfo.error_write") + '\n');
                }
            }
        } catch (Exception ignored) {
        }
    }

    /**
     * Gets input from the command line.
     * <p/>
     * If the blank string is allowed, then add it to the <code>expected</code> array.
     *
     * @param output   The label to print
     * @param expected List of possible values
     * @return One of the expected values
     */
    private static String getInput(String output, String[] expected) {

        String input;

        while (true) {
            System.out.print(output);
            BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
            try {
                input = br.readLine();
                if (expected != null && expected.length >= 1) {
                    for (int i = 0; i < expected.length; i++) {
                        if (input.equals(expected[i])) return input;
                    }
                } else return input;  // Accept anything, if expected is not set
            } catch (Exception ignored) {
            }
        }
    }

    /**
     * Implements the GUI-based mode of this application.
     */
    private static void processGUI() {

        String name = JavaInfo.class.getName();  // getSimpleName() is still being invented ;-)
        name = name.substring(name.lastIndexOf('.') + 1);
        String title = name;
        Properties props = new Properties();  // Avoid using JavaInfoSettings (Java 1.4)!
        try {
            props.load(JavaInfo.class.getResourceAsStream(PROPSPATH));
            title = props.getProperty("APPLICATION") + ' ' + props.getProperty("VERSION");
            if (System.getProperty("SERVLET_URL") != null) {
                SERVLET_URL = System.getProperty("SERVLET_URL");
            } else {
                SERVLET_URL = props.getProperty("SERVLET_URL");
            }
            POLICY_URL = props.getProperty("POLICY_URL");
        } catch (Exception ignored) {
        }

        // Construct the Swing window with the few components
        JavaInfoView view = new JavaInfoView(props);
        view.showWindow(title);
    }

    /**
     * Creates a normalized representation for the properties.
     *
     * @param props The container with properties
     * @return Binary data, <code>null</code> on error
     */
    public static byte[] normalizeProps(Properties props) {

        String key, line;
        byte[] binaryline;
        ByteArrayOutputStream bstream = new ByteArrayOutputStream();
        Collator collator = Collator.getInstance(Locale.US);
        collator.setStrength(Collator.PRIMARY);

        try {
            // Assemble the binary data from normalized properties. The lines in the format
            // "key=value" are sorted alphabetically with a Vector and calculated. We use
            // a fixed encoding for the conversion from char (String) to byte. The sorting
            // must be done on-the-fly since Collections.sort() isn't available for Java 1.4

            Vector sorter = JavaInfoUtils.sort(props);

            int size = props.size();
            for (int i = 0; i < size; i++) {
                key = ((String) sorter.elementAt(i)).toLowerCase();  // :-)
                line = key + '=' + props.getProperty(key, "").toUpperCase() + '\n';
                binaryline = line.getBytes(JavaInfoUtils.ENCODING);
                bstream.write(binaryline);
            }
            bstream.flush();

            return bstream.toByteArray();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Prints the usage message via <code>System.out</code>.
     */
    private static void printUsage() {

        String name = JavaInfo.class.getName();  // getSimpleName() is still being invented
        name = name.substring(name.lastIndexOf('.') + 1).toLowerCase();

        String sep = System.getProperties().getProperty("line.separator");
        if (sep == null) sep = "\n";

        String msg = getLocalizedText("JavaInfo.usage");
        msg = new String(JavaInfoUtils.replace(msg.toCharArray(), "%NAME%", name));
        msg = new String(JavaInfoUtils.replace(msg.toCharArray(), "\n", sep));
        System.out.println(msg);
        JavaInfo.printCopyright();
    }

    /**
     * Prints the copyright message via <code>System.out</code>.
     *
     * @since 1.5
     */
    private static void printCopyright() {
        System.out.println(JavaInfoUtils.COPYRIGHT);
    }

    /**
     * Prints the version message via <code>System.out</code>.
     *
     * @since 1.5
     */
    private static void printVersion() {

        // We recycle the global resource file 'deploy.properties'

        try {
            Properties props = new Properties();
            props.load(JavaInfo.class.getResourceAsStream(JavaInfo.PROPSPATH));

            String sep = System.getProperties().getProperty("line.separator");
            if (sep == null) sep = "\n";

            String version = props.getProperty("VERSION");
            String build = props.getProperty("BUILD");

            String date = props.getProperty("DATE");
            String pattern = props.getProperty("DATE_PATTERN");
            String locale = props.getProperty("DATE_LOCALE");

            SimpleDateFormat formatter = new SimpleDateFormat(pattern, new Locale(locale, ""));
            Date dateobj = formatter.parse(date, new ParsePosition(0));

            // Convert date to ISO format (yyyy-MM-dd)
            formatter = new SimpleDateFormat("yyyy-MM-dd", Locale.US);
            String isodate = formatter.format(dateobj);

            String name = JavaInfo.class.getName();  // getSimpleName() is still being invented
            name = name.substring(name.lastIndexOf('.') + 1);
            String fullversion = name + ' ' + version + " Build " + build + " (" + isodate + ')';

            System.out.println(sep + fullversion + sep);
            JavaInfo.printCopyright();
        } catch (Exception ignored) {
        }
    }

    /**
     * Gets the localized text from the resource bundle associated with this class.
     *
     * @param name The name of the variable
     * @return The text, <code>null</code> if the variable does not exist
     */
    protected static String getLocalizedText(String name) {

        // Consults the default locale for this
        ResourceBundle bundle = ResourceBundle.getBundle(JavaInfo.class.getName(), new UTF8Control());
        return bundle.getString(name);
    }

    /**
     * This can be used for the application be executed standalone.
     * <p>
     * Supported arguments:
     * <ul>
     * <li><code>-locale [value]</code>      The locale to use (en|de), automatically set otherwise</li>
     * <li><code>-help or -?</code>          Print some help messages</li>
     * <li><code>-version</code>             Print the current version</li>
     * <li><code>-stdout</code>              Write properties to stdout directly w/o GUI</li>
     * <li><code>-headless</code>            Optional flag to run application in headless mode</li>
     * </ul>
     *
     * @param args Command line arguments
     */
    public static void main(String[] args) {

        boolean headless = false;
        try {
            Properties props = new Properties();  // Avoid using JavaInfoSettings (Java 1.4)!
            props.load(JavaInfo.class.getResourceAsStream(PROPSPATH));
            JAVAINFO_APP_NAME = props.getProperty("APPLICATION") + ' ' + props.getProperty("VERSION");
        } catch (Exception ignored) {
        }
        try {
            // Evaluate the command line parameters

            // Step 1: Determine the locale to print the message as expected
            for (int i = 0; i < args.length; i++) {

                if (args[i].equals("-locale")) {

                    // Check out value and set system property accordingly.
                    // We cannot use the ISO 639-2 codes due to Java 1.4 compatibility.
                    if ((i + 1) < args.length) {
                        String locale = args[i + 1];
                        if (locale.equals("en")) Locale.setDefault(Locale.ENGLISH);
                        else if (locale.equals("de")) {
                            Locale.setDefault(Locale.GERMAN);
                            JavaInfoUtils.LF_LOCALE = "ger_DE";  // Web service uses ISO 639-2
                        }
                        // Default is the system default for the user, not "user.language"!
                    }
                }
            }
            if (Locale.getDefault().getLanguage().equals("de")) JavaInfoUtils.LF_LOCALE = "ger_DE";

            // Step 2: Do what the user likes us to
            for (int i = 0; i < args.length; i++) {

                if (args[i].equals("-help") || args[i].equals("-?")) {
                    JavaInfo.printUsage();
                    return;
                } else if (args[i].equals("-version")) {
                    JavaInfo.printVersion();
                    return;
                } else if (args[i].equals("-stdout")) {

                    // Continue without GUI, write properties to stdout directly
                    byte[] data = JavaInfoUtils.generateData(JAVAINFO_APP_NAME);
                    System.out.write(data);
                    return;
                } else if (args[i].equals("-headless")) headless = true;
            }

            if (headless || GraphicsEnvironment.isHeadless()) {
                JavaInfo.processCLI();
            } else JavaInfo.processGUI();
        } catch (Exception ignored) {
        }
    }
}