/*
    Copyright (c) 2005-2025 Leisenfels GmbH GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.javainfo.gui;

import javax.swing.*;
import javax.swing.border.EmptyBorder;
import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileOutputStream;
import java.net.URI;
import java.net.URL;
import java.util.Properties;


/**
 * Main graphical component.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.8
 */
public class JavaInfoView extends JFrame {


    /**
     * The container holding some config values.
     */
    protected Properties props;


    /**
     * Constructor method for the manifest display.
     *
     * @param props Some config values
     */
    public JavaInfoView(Properties props) {

        super();
        this.props = props;

        initComponents();
        if (java.beans.Beans.isDesignTime()) return;
    }

    /**
     * Prints the information from MANIFEST.MF via <code>System.out</code>.
     *
     * @param title The window title
     */
    protected void showWindow(String title) {

        // Then set the component localization
        this.chkAgree.setText(JavaInfo.getLocalizedText("JavaInfo.agree"));
        this.btnPolicy.setText(JavaInfo.getLocalizedText("JavaInfo.policy"));
        this.btnSubmit.setText(JavaInfo.getLocalizedText("JavaInfo.submit"));
        this.btnSave.setText(JavaInfo.getLocalizedText("JavaInfo.saveas"));
        this.btnExit.setText(JavaInfo.getLocalizedText("JavaInfo.exit"));

        // Create the frame
        this.setTitle(title);
        URL url = JavaInfo.class.getResource(JavaInfo.ICONPATH);
        Image image = Toolkit.getDefaultToolkit().getImage(url);
        this.setIconImage(image);
        this.setContentPane(this.pnlMain);
        this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        this.pack();  // Necessary for Linux

        Toolkit toolkit = this.getToolkit();
        Dimension dims = toolkit.getScreenSize();

        // Place the window in the center of the screen or use user settings
        int defaultWidth = 580;
        int defaultHeight = 420;
        int defaultX = (int) ((dims.width - defaultWidth) / 2.0); // Center
        int defaultY = (int) ((dims.height - defaultHeight) / 2.0) - 10;
        this.setSize(defaultWidth, defaultHeight);
        this.setLocation(defaultX, defaultY);

        JavaInfoView.setMessage(this.lblStatus, "", true, 1);

        String copyright = JavaInfoUtils.COPYRIGHT.replaceFirst("([(]c[)])", "&copy;");
        this.lblCopyright.setText("<html>&nbsp;<i><font size='2'>" + copyright + "</font></i></html>");

        // Add the normalized system properties finally
        final byte[] data = JavaInfoUtils.generateData(JavaInfo.JAVAINFO_APP_NAME);
        try {
            taProps.setText(new String(data, JavaInfoUtils.ENCODING));
        } catch (Exception ignored) {
        }

        // React on user clicks
        addWindowListener(new WindowAdapter() { // Window "X"
            public void windowClosing(WindowEvent event) {
                System.exit(0);
            }
        });

        this.btnExit.addActionListener(event -> System.exit(0));

        this.btnSave.addActionListener(event -> new Thread(new Runnable() {  // Do not block the GUI

            public void run() {

                // Present chooser for target directory
                JFileChooser chooser = new JFileChooser();
                // Do not adjust the user dir here, looks strange on Mac OS X.
                // Normally, the file chooser automagically navigates to the user's home.

                chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
                chooser.setMultiSelectionEnabled(false);
                chooser.setSelectedFile(new File("props.xml"));
                chooser.setDialogTitle(JavaInfo.getLocalizedText("JavaInfo.saveas"));
                Dimension dims1 = chooser.getSize();
                chooser.setLocation(centerForWindow(JavaInfoView.this, dims1.width, dims1.height));

                int retval = chooser.showSaveDialog(JavaInfoView.this);
                if (retval != JFileChooser.APPROVE_OPTION) {
                    return;  // Cancelled
                }

                File target = chooser.getSelectedFile();
                try {
                    FileOutputStream fstream = new FileOutputStream(target);
                    fstream.write(data);
                    fstream.flush();
                    fstream.close();
                    setMessage(lblStatus, JavaInfo.getLocalizedText("JavaInfo.ok_write"), true, 10000);
                } catch (Exception e) {
                    setMessage(lblStatus, JavaInfo.getLocalizedText("JavaInfo.error_write"), false, 10000);
                }
            }
        }, "[javainfo] File Saver").start());

        this.btnSubmit.addActionListener(event -> {

            // Don't freeze the EDT, submission may take some time or may block!
            final Thread thread = new Thread(() -> {

                btnSubmit.setEnabled(false);
                btnSave.setEnabled(false);
                String status = null;

                try {
                    URL url1 = new URL(JavaInfo.SERVLET_URL + "?locale=" + JavaInfoUtils.LF_LOCALE);
                    byte[] result = JavaInfoUtils.callWebServiceByPOST(url1, data);

                    // Status OK? If not, the service possibly returned an error message
                    status = JavaInfoUtils.checkResponse(result);
                } catch (Exception e) {
                    if (e instanceof InterruptedException) {
                        status = JavaInfo.getLocalizedText("JavaInfo.timeout");
                    }
                } finally {
                    btnSubmit.setEnabled(true);
                    btnSave.setEnabled(true);

                    if (status != null && status.equals("OK")) {
                        String text = JavaInfo.getLocalizedText("JavaInfo.ok_submit");
                        setMessage(lblStatus, text, true, 10000);
                    } else {
                        if (status != null) {
                            setMessage(lblStatus, status, false, 10000);
                        } else {
                            status = JavaInfo.getLocalizedText("JavaInfo.error_submit");
                            setMessage(lblStatus, status, false, 10000);
                        }
                    }
                }
            }, "[javainfo] Submitter");
            thread.start();

            // And another thread to monitor the first thread
            new Thread(() -> {
                try {
                    thread.join(10000);  // Wait max. 10 seconds, then interrupt it
                } catch (Exception ignored) {
                }
                try {
                    if (thread.isAlive()) {
                        thread.interrupt();
                    }
                } catch (Exception ignored) {
                }
            }, "[javainfo] Submitter Monitor").start();
        });

        this.btnPolicy.addActionListener(event -> {
            try {
                // Let the system open the default Internet browser
                if (Desktop.isDesktopSupported()) Desktop.getDesktop().browse(new URI(JavaInfo.POLICY_URL));
            } catch (Exception ignored) {
            }
        });

        this.chkAgree.addActionListener(event -> {
            // Submit is only possible if the user agreed the Privacy Policy
            this.btnSubmit.setEnabled(this.chkAgree.isSelected());
        });

        this.setVisible(true);
    }

    /**
     * Calculates the position for a dialog window on the screen.
     * <p/>
     * Can be used to center a dialog for its parent window. If the parent window is <code>null</code>
     * then the point is calculated for the whole desktop automatically.
     *
     * @param parent The parent window to center the dialog, for desktop if <code>null</code>
     * @param width  The width of the dialog to be centered
     * @param height The height of the dialog to be centered
     * @return The position on the screen
     */
    public static Point centerForWindow(Window parent, int width, int height) {

        // Calculate the parent center, use desktop if parent is not available
        Toolkit toolkit = Toolkit.getDefaultToolkit();
        Dimension size = parent != null ? parent.getSize() : toolkit.getScreenSize();
        Point pos = parent != null ? parent.getLocation() : new Point(0, 0);
        int centerX = pos.x + size.width / 2;
        int centerY = pos.y + size.height / 2;

        // Now calculate the dialog upper left corner
        int locationX = Math.max(0, centerX - width / 2);
        int locationY = Math.max(0, centerY - height / 2);

        return new Point(locationX, locationY);
    }

    /**
     * Prints a message for the user.
     *
     * @param label   The component
     * @param text    The text
     * @param success Successful?
     * @param millis  Shown for n milliseconds, nonstop if -1
     */
    private static void setMessage(final JLabel label, String text, boolean success, final int millis) {

        if (success) label.setForeground(Color.green.darker());
        else label.setForeground(Color.red);

        final int chars = 60;
        StringBuilder textBuilder = new StringBuilder(text);
        while (textBuilder.length() < chars) textBuilder.append(" ");
        text = textBuilder.toString();
        label.setText(text);

        new Thread(new Runnable() {
            public void run() {
                try {
                    if (millis >= 0) Thread.sleep(millis);
                    StringBuilder text = new StringBuilder();
                    while (text.length() < chars) text.append(" ");
                    label.setText(text.toString());
                } catch (Exception ignored) {
                }
            }
        }, "[javainfo] Message Setter").start();
    }

    private void initComponents() {
        // JFormDesigner - Component initialization - DO NOT MODIFY  //GEN-BEGIN:initComponents
        pnlMain = new JPanel();
        JScrollPane scrProps = new JScrollPane();
        taProps = new JTextArea();
        JPanel panel2 = new JPanel();
        JPanel panel3 = new JPanel();
        lblCopyright = new JLabel();
        panel5 = new JPanel();
        chkAgree = new JCheckBox();
        toolBar1 = new JToolBar();
        btnPolicy = new JButton();
        JPanel panel1 = new JPanel();
        lblStatus = new JLabel();
        JPanel panel4 = new JPanel();
        btnSubmit = new JButton();
        btnSave = new JButton();
        btnExit = new JButton();
        lblWest = new JLabel();
        lblEast = new JLabel();
        lblNorth = new JLabel();

        //======== pnlMain ========
        {
            pnlMain.setLayout(new BorderLayout(5, 0));

            //======== scrProps ========
            {

                //---- taProps ----
                taProps.setEnabled(true);
                taProps.setEditable(false);
                scrProps.setViewportView(taProps);
            }
            pnlMain.add(scrProps, BorderLayout.CENTER);

            //======== panel2 ========
            {
                panel2.setBorder(new EmptyBorder(0, 5, 5, 0));
                panel2.setLayout(new BorderLayout());

                //======== panel3 ========
                {
                    panel3.setLayout(new BorderLayout());

                    //---- lblCopyright ----
                    lblCopyright.setText("Copyright");
                    lblCopyright.setVerticalAlignment(SwingConstants.TOP);
                    panel3.add(lblCopyright, BorderLayout.NORTH);

                    //======== panel5 ========
                    {
                        panel5.setLayout(new FlowLayout(FlowLayout.RIGHT));

                        //---- chkAgree ----
                        chkAgree.setText("I agree");
                        panel5.add(chkAgree);

                        //======== toolBar1 ========
                        {
                            toolBar1.setFloatable(false);

                            //---- btnPolicy ----
                            btnPolicy.setText("Open...");
                            toolBar1.add(btnPolicy);
                        }
                        panel5.add(toolBar1);
                    }
                    panel3.add(panel5, BorderLayout.CENTER);

                    //======== panel1 ========
                    {
                        panel1.setLayout(new BorderLayout());
                        panel1.add(lblStatus, BorderLayout.CENTER);

                        //======== panel4 ========
                        {
                            panel4.setLayout(new FlowLayout());

                            //---- btnSubmit ----
                            btnSubmit.setText("Submit");
                            btnSubmit.setVerticalAlignment(SwingConstants.BOTTOM);
                            btnSubmit.setEnabled(false);
                            panel4.add(btnSubmit);

                            //---- btnSave ----
                            btnSave.setText("Save As...");
                            btnSave.setVerticalAlignment(SwingConstants.BOTTOM);
                            panel4.add(btnSave);

                            //---- btnExit ----
                            btnExit.setText("Exit");
                            btnExit.setVerticalAlignment(SwingConstants.BOTTOM);
                            panel4.add(btnExit);
                        }
                        panel1.add(panel4, BorderLayout.EAST);
                    }
                    panel3.add(panel1, BorderLayout.SOUTH);
                }
                panel2.add(panel3, BorderLayout.CENTER);
            }
            pnlMain.add(panel2, BorderLayout.SOUTH);
            pnlMain.add(lblWest, BorderLayout.WEST);
            pnlMain.add(lblEast, BorderLayout.EAST);
            pnlMain.add(lblNorth, BorderLayout.NORTH);
        }
        // JFormDesigner - End of component initialization  //GEN-END:initComponents
    }

    // JFormDesigner - Variables declaration - DO NOT MODIFY  //GEN-BEGIN:variables
    private JPanel pnlMain;
    private JTextArea taProps;
    private JLabel lblCopyright;
    private JPanel panel5;
    private JCheckBox chkAgree;
    private JToolBar toolBar1;
    private JButton btnPolicy;
    private JLabel lblStatus;
    private JButton btnSubmit;
    private JButton btnSave;
    private JButton btnExit;
    private JLabel lblWest;
    private JLabel lblEast;
    private JLabel lblNorth;
    // JFormDesigner - End of variables declaration  //GEN-END:variables

}
