/*
    Copyright (c) 2005-2026 Leisenfels GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.javainfo.gui;

import javax.net.ssl.*;
import java.io.*;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLConnection;
import java.security.MessageDigest;
import java.security.cert.X509Certificate;
import java.text.Collator;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Properties;
import java.util.Vector;


/**
 * This class holds various methods for common use with JavaInfo.
 * <p/>
 * Make sure that this class can also be used by the Android classes. References to the JavaInfo
 * class must be avoided since Applet is not part of the Android API for example.
 *
 * @author Axel Schwolow
 * @since 1.5
 */
public class JavaInfoUtils {


    /**
     * The copyright information.
     */
    public final static String COPYRIGHT = "JavaInfo (c) 2005-2026 Leisenfels GmbH. All rights reserved.";
    /**
     * The property name holding the checksum value.
     */
    public final static String CHECKSUMPROP = "com.lf.javainfo.gui.JavaInfo.checksum";
    /**
     * Globally used encoding/charset.
     */
    public final static String ENCODING = "UTF-8";
    /**
     * The locale to call the web service with (eng_US|ger_DE).
     */
    public static String LF_LOCALE = "eng_US";
    /**
     * The property name holding the response status.
     */
    public final static String STATUSPROP = "com.lf.javainfo.gui.JavaInfo.status";

    /**
     * Constructor method for i18n purposes only.
     * <p/>
     * Can only be used if beans design time is set (e.g. by external tools like the
     * <code>I18NExtractor</code>).
     * <p/>
     *
     * @throws InstantiationException Error indication
     * @since 1.6
     */
    public JavaInfoUtils() throws InstantiationException {

        boolean designtime = false;
        try {
            // Android and Java 1.1
            Class clazz = Class.forName("java.beans.Beans");
            Method[] methods = clazz.getMethods();
            for (int i = 0; i < methods.length; i++) {
                if (methods[i].getName().equals("isDesignTime")) {
                    designtime = ((Boolean) methods[i].invoke((Object) null, (Object[]) null)).booleanValue();
                    break;
                }
            }
        } catch (Exception ignored) {
        }
        if (!designtime) throw new InstantiationException("Do not use this constructor!");
    }

    /**
     * Replaces all occurrencies of the pattern string by a certain sequence.
     * <p/>
     * For the example string "Hamburger" the call with
     * <code>pattern="burg"</code> and <code>replacement="m"</code> will
     * return "Hammer".
     * <p/>
     *
     * @param array       The char array to replace the sequence
     * @param pattern     The string to search for
     * @param replacement This is used for pattern substitution
     * @return The processed string or <code>null</code> if an error occurred
     * @since 1.6
     */
    public static char[] replace(char[] array, String pattern, String replacement) {

        char[] result;
        char[] substring;
        int i;
        int k;
        int chars = 0;
        int pos = 0;


        if (array == null || pattern == null || replacement == null) return null;
        if (pattern.isEmpty()) return array;

        // Let's start replacing characters
        Vector collect = new Vector(0);
        for (i = 0; i <= (array.length - pattern.length()); i++) {

            substring = JavaInfoUtils.substring(array, i, (i + pattern.length() - 1));
            if (JavaInfoUtils.startsWith(substring, pattern)) {

                // Found replacement candidate!
                collect.addElement(replacement.toCharArray());
                chars = chars + replacement.length();

                // And skip the original characters
                i = i + pattern.length() - 1;
            } else {
                // Add the next character to the collector
                substring = new char[1];
                substring[0] = array[i];
                collect.addElement(substring);
                chars++;
            }
        }

        // There may be an unprocessed rest, set it
        if (i > (array.length - pattern.length()) && i < array.length) {
            substring = JavaInfoUtils.substring(array, i, array.length - 1);
            collect.addElement(substring);
            chars = chars + substring.length;
        }

        // Finally we have to assemble the result char array
        result = new char[chars];
        int size = collect.size();
        for (i = 0; i < size; i++) {
            substring = (char[]) collect.elementAt(i);
            for (k = 0; k < substring.length; k++) {
                result[pos] = substring[k];
                pos++;
            }
        }
        collect.removeAllElements();

        return result;
    }

    /**
     * Extracts a sub-string from a given <code>char[]</code>
     * (like <code>String</code>).
     * <p/>
     * For the example string "Hamburger" the call with <code>begin=3</code>
     * and <code>end=6</code> will return "burg".
     * <p/>
     *
     * @param array The char array to check for the suffix
     * @param begin Index to start building the substring
     * @param end   Index to stop building the substring
     * @return The substring or <code>null</code> if error occurred
     * @since 1.6
     */
    public static char[] substring(char[] array, int begin, int end) {

        if (array == null) return null;
        if (begin < 0 || begin > (array.length - 1)) return null;
        if (end < 0 || end > (array.length - 1)) return null;
        if (end < begin) return null;

        // Let's generate the substring then
        char[] substring = new char[end - begin + 1];
        for (int i = begin; i <= end; i++) {
            substring[i - begin] = array[i];
        }
        return substring;
    }

    /**
     * Checks whether a given char array starts with a given <code>String</code>.
     * <p/>
     * Returns <code>false</code> for empty prefix strings with any array!
     * <p/>
     *
     * @param array  The char array to check for the prefix
     * @param prefix The prefix string be detected
     * @return Does the char array start with the prefix?
     * @since 1.6
     */
    public static boolean startsWith(char[] array, String prefix) {

        if (array == null || prefix == null) return false;
        if (prefix.isEmpty()) return false;
        char[] prefixarray = prefix.toCharArray();

        // Let's compare it char for char
        if (array.length >= prefixarray.length) {
            for (int i = 0; i < prefixarray.length; i++) {
                if (prefixarray[i] != array[i]) return false;
            }
        } else return false;
        return true;
    }

    /**
     * Translates any element/attribute content to proper XML (including default entities).
     * <p/>
     * See class <code>XMLUtils</code> for details.
     * <p/>
     *
     * @param target      The string to be translated for XML use
     * @param replacement Unallowed characters like 0x0 are replaced, or deleted if <code>null</code>
     * @return The XML conforming translation
     */
    public static String translateToXMLContent(String target, String replacement) {

        // Empty string is returned promptly
        if (target.isEmpty()) return target;

        String result = "";  // StringBuilder not available...
        char[] array = target.toCharArray();
        for (int i = 0; i < array.length; i++) {

            if (array[i] == 0x9 || array[i] == 0xa || array[i] == 0xd ||
                    (array[i] >= 0x20 && array[i] <= 0xd7ff) ||
                    (array[i] >= 0xe000 && array[i] <= 0xfffd) ||
                    (array[i] >= 0x10000 && array[i] <= 0x10ffff)) {

                // Reserved XML character to be substituted?
                if (array[i] == '<') result += "&lt;";
                else if (array[i] == '>') result += "&gt;";
                else if (array[i] == '&') result += "&amp;";
                else if (array[i] == '\'') result += "&apos;";
                else if (array[i] == '"') result += "&quot;";
                else result += array[i];  // Allowed by the XML spec
            } else {  // All others are not allowed as XML content
                if (replacement != null) result += replacement;
            }
        }
        return result;
    }

    /**
     * Converts a single byte to hex representation.
     * <p/>
     *
     * @param b         Data to be converted
     * @param uppercase Use uppercase hex characters, lowercase if <code>false</code>
     * @return Calculated hex representation
     */
    public static String convertToHex(byte b, boolean uppercase) {

        int value = (b & 0x7F) + (b < 0 ? 128 : 0);
        String ret = (value < 16 ? "0" : "");
        ret += uppercase ? Integer.toHexString(value).toUpperCase() : Integer.toHexString(value);
        return ret;
    }

    /**
     * Sorts the properties keys.
     * <p/>
     *
     * @param props The container with properties
     * @return The container with sorted properties, <code>null</code> on error
     * @since 1.6
     */
    public static Vector sort(Properties props) {

        String key, sorterkey;
        Collator collator = Collator.getInstance(Locale.US);
        collator.setStrength(Collator.PRIMARY);

        try {
            int size = props.size();
            Vector sorter = new Vector(size);

            Enumeration en = props.keys();
            while (en.hasMoreElements()) {
                key = (String) en.nextElement();

                int count = sorter.size();
                for (int i = 0; i < count; i++) {
                    sorterkey = (String) sorter.elementAt(i);
                    if (collator.compare(sorterkey, key) > 0) {
                        sorter.insertElementAt(key, i);
                        break;
                    }
                }
                if (count == sorter.size()) sorter.addElement(key);
            }
            return sorter;
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Sends a message via HTTP POST to a web service URL.
     * <p/>
     * IMPORTANT: If Tomcat calls doGet() is called instead of doPost() you have to include
     * the trailing slash at the end of the URL. Tomcat does some buggy address mapping.
     * <p/>
     * Found the solution here:
     * <p/>
     * <a href="http://bytes.com/groups/java/17880-using-httpurlconnection-post-servlet-tomcat-5-results-get-not-post-server-side">...</a>
     * <p/>
     *
     * @param url  Web service URL, include trailing slash for doPost() to be called
     * @param data Payload of POST request
     * @return Payload of received message, <code>null</code> if error occurred
     */
    public static byte[] callWebServiceByPOST(URL url, byte[] data) {

        try {
            // We use HTTPS, ignore certificate errors like e.g. javax.net.ssl.SSLHandshakeException
            TrustManager[] trustAllCerts = new TrustManager[]{new X509TrustManager() {
                public X509Certificate[] getAcceptedIssuers() {
                    return null;
                }

                public void checkClientTrusted(X509Certificate[] certs, String authType) {
                }

                public void checkServerTrusted(X509Certificate[] certs, String authType) {
                }
            }};

            SSLContext sc = SSLContext.getInstance("SSL");
            sc.init(null, trustAllCerts, new java.security.SecureRandom());
            HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());

            // Create all-trusting host name verifier
            HostnameVerifier allHostsValid = (hostname, session) -> true;
            // Install the all-trusting host verifier
            HttpsURLConnection.setDefaultHostnameVerifier(allHostsValid);

            // Open connection and configure HTTP POST method
            URLConnection urlConn = url.openConnection();
            urlConn.setRequestProperty("Content-Type", "text/xml");
            urlConn.setRequestProperty("Content-Length", String.valueOf(data.length));
            urlConn.setUseCaches(false);
            urlConn.setDoOutput(true);
            urlConn.setDoInput(true);

            // Send the request
            OutputStream ostream = urlConn.getOutputStream();
            ostream.write(data);
            ostream.flush();
            ostream.close();

            // Receive the response
            InputStream istream = urlConn.getInputStream();
            ByteArrayOutputStream bstream = new ByteArrayOutputStream();

            int len;
            byte[] buffer = new byte[1024];
            while ((len = istream.read(buffer)) != -1) {
                bstream.write(buffer, 0, len);
            }
            istream.close();
            bstream.close();
            return bstream.toByteArray();
        } catch (Exception e) {
            System.err.println(e);
        }
        return null;
    }

    /**
     * Assembles the data array with system properties including a header and a checksum.
     * <p/>
     *
     * @param appname Name of the app like "JavaInfo21"
     * @return Array with data in the global encoding
     * @since 1.6
     */
    public static byte[] generateData(String appname) {

        String sep = System.getProperties().getProperty("line.separator");
        if (sep == null) sep = "\n";

        ByteArrayOutputStream bstream = new ByteArrayOutputStream();
        Properties props = System.getProperties();
        int size = props.size();

        // Prepare the header
        String header = "    Java system properties generated with " + appname + "." + sep;  // English
        header += "    " + JavaInfoUtils.COPYRIGHT;

        // Calculate the checksum for verification to be included in the header.
        // We let the Properties class create a byte array for this purpose.
        // Manipulations on the data can be detected easily with this checksum.
        String checksum = calculateChecksum(props);

        // Add our checksum directly as property (class in FQN should be unique)
        props.put(JavaInfoUtils.CHECKSUMPROP, checksum != null ? checksum : "");
        // Java 1.1: no setProperty()

        try {
            // Assemble XML file based on 'parameter_message.dtd'
            bstream.write(("<?xml version=\"1.0\" encoding=\"" + JavaInfoUtils.ENCODING + "\"?>" + sep).getBytes(JavaInfoUtils.ENCODING));
            bstream.write(("<!--" + sep).getBytes(JavaInfoUtils.ENCODING));
            bstream.write((header + sep).getBytes(JavaInfoUtils.ENCODING));
            bstream.write(("  -->" + sep).getBytes(JavaInfoUtils.ENCODING));
            bstream.write(("<parameter_message count=\"" + size + "\">" + sep).getBytes(JavaInfoUtils.ENCODING));

            Vector sorter = sort(props);
            size = sorter.size();
            for (int i = 0; i < size; i++) {
                String key = translateToXMLContent((String) sorter.elementAt(i), "?");
                String value = translateToXMLContent(props.getProperty(key), "?");
                bstream.write(("    <parameter name=\"" + key + "\">" + value + "</parameter>" + sep).getBytes(JavaInfoUtils.ENCODING));
            }
            bstream.write(("</parameter_message>" + sep).getBytes(JavaInfoUtils.ENCODING));
        } catch (Exception ignored) {
        }
        return bstream.toByteArray();
    }

    /**
     * Calculates a simple checksum for normalized properties.
     * <p/>
     *
     * @param props The container with properties
     * @return Checksum, <code>null</code> on error
     * @since 1.6
     */
    public static String calculateChecksum(Properties props) {

        String key, line;
        byte[] binaryline;
        ByteArrayOutputStream bstream = new ByteArrayOutputStream();
        Collator collator = Collator.getInstance(Locale.US);
        collator.setStrength(Collator.PRIMARY);

        try {
            // Assemble the binary data from normalized properties. The lines in the format
            // "key=value" are sorted alphabetically with a Vector and calculated. We use
            // a fixed encoding for the conversion from char (String) to byte. The sorting
            // must be done on-the-fly since Collections.sort() isn't available for Java 1.1

            Vector sorter = sort(props);

            int size = props.size();
            for (int i = 0; i < size; i++) {
                key = (String) sorter.elementAt(i);
                line = key + '=' + props.getProperty(key) + '\n';
                binaryline = line.getBytes(JavaInfoUtils.ENCODING);
                bstream.write(binaryline);
            }
            bstream.flush();

            // And calculate the MD5 message digest
            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(bstream.toByteArray());
            byte[] digest = md.digest();
            StringBuffer checksum = new StringBuffer();
            for (int i = 0; i < digest.length; ++i) {
                checksum.append(convertToHex(digest[i], false));  // Lowercase
            }
            return checksum.toString();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Extracts the result of the submission.
     * <p/>
     *
     * @param data The response data
     * @return Status from the web service ("OK" = success or error message or <code>null</code>)
     */
    public static String checkResponse(byte[] data) {

        String result = null;

        try {
            // We must parse the XML data manually since Java 1.1 lacks XML support
            String line;
            ByteArrayInputStream bistream = new ByteArrayInputStream(data);
            InputStreamReader isreader = new InputStreamReader(bistream, ENCODING);
            LineNumberReader lnreader = new LineNumberReader(isreader);

            while ((line = lnreader.readLine()) != null) {
                line = line.trim();

                // Search for OK message pattern ('parameter_message')
                if (line.equals("<parameter name=\"" + STATUSPROP + "\">OK</parameter>")) {
                    result = "OK";
                    break;
                } else if (line.startsWith("<description>")) {  // Error message
                    result = line.substring(13, line.length() - 14);
                    break;
                }
            }
            bistream.close();

            /*
           <?xml version="1.0" encoding="ISO-8859-1"?>
           <parameter_message version="1.0" count="1">
             <parameter name="com.lf.javainfo.gui.JavaInfo.status">OK</parameter>
           </parameter_message>

           <?xml version="1.0" encoding="ISO-8859-1"?>
           <error_message version="1.0" locale="eng_US">
             <status code="10">Data error</status>
             <description>Properties already exist</description>
           </error_message>
            */
        } catch (Exception ignored) {
        }
        return result;
    }
}
