/*
    Copyright (c) 2005-2025 Leisenfels GmbH GmbH GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.commons.info;

import javax.swing.*;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;


/**
 * Main graphical component.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.4
 */
public class JarInfoView extends JFrame implements ListSelectionListener {


    /**
     * The main class with some utility methods.
     */
    protected JarInfo jarInfo;
    /**
     * The tree root node.
     */
    protected DefaultMutableTreeNode rootNode = null;
    /**
     * Model of the unit list.
     */
    protected DefaultTreeModel treeModel = null;


    /**
     * Constructor method for the manifest display.
     *
     * @param jarInfo The model holding some config values
     */
    public JarInfoView(JarInfo jarInfo) {

        super();
        this.jarInfo = jarInfo;
        initComponents();
        if (java.beans.Beans.isDesignTime()) return;
    }

    /**
     * Prints the information from MANIFEST.MF via <code>System.out</code>.
     *
     * @param jarFile The path of the JAR file
     * @param bundle  The string resources
     */
    protected void showWindow(final String jarFile, final ResourceBundle bundle) {

        ZipEntry next;

        // Then set the component localization
        this.btnExtract.setText(bundle.getString("JarInfo.btnExtract.text"));
        this.btnClose.setText(bundle.getString("JarInfo.btnClose.text"));
        this.tabbedPane.setTitleAt(this.tabbedPane.indexOfComponent(this.tabNotice), bundle.getString("JarInfo.tabNotice.title"));
        this.tabbedPane.setTitleAt(this.tabbedPane.indexOfComponent(this.tabManifest), bundle.getString("JarInfo.tabManifest.title"));
        this.tabbedPane.setTitleAt(this.tabbedPane.indexOfComponent(this.tabEntries), bundle.getString("JarInfo.tabEntries.title"));
        this.tabbedPane.setTitleAt(this.tabbedPane.indexOfComponent(this.tabServices), bundle.getString("JarInfo.tabServices.title"));

        // Create the frame
        this.setContentPane(this.pnlMain);
        this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        JarInfoView.setMessage(this.lblStatus, "", null, 1);

        String clazz = JarInfo.class.getName();
        clazz = clazz.substring(clazz.lastIndexOf('.') + 1);
        String proptitle = this.jarInfo.props.getProperty("JAVAINFO_WINDOW_TITLE");
        this.setTitle(proptitle != null && proptitle.trim().length() >= 1 ? proptitle : clazz);

        try {
            // Try to load configured icon, Java cup is displayed if not set (default)
            String iconpath = this.jarInfo.props.getProperty("JAVAINFO_WINDOW_ICON", "");
            this.setIconImage(new ImageIcon(this.getClass().getResource(iconpath)).getImage());
        } catch (Exception ignored) {
        }

        this.btnClose.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent event) {
                System.exit(0);
            }
        });

        this.btnExtract.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent event) {

                new Thread(new Runnable() {  // Do not block the GUI

                    public void run() {

                        // Present chooser for target directory
                        JFileChooser chooser = new JFileChooser();
                        // Do not adjust the user dir here, looks strange on Mac OS X.
                        // Normally, the file chooser automagically navigates to the user's home.

                        chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
                        chooser.setMultiSelectionEnabled(false);
                        chooser.setDialogTitle(bundle.getString("JarInfo.chooser.title"));
                        Dimension dims = chooser.getSize();
                        chooser.setLocation(centerForWindow(JarInfoView.this, dims.width, dims.height));

                        int retval = chooser.showSaveDialog(JarInfoView.this);
                        if (retval != JFileChooser.APPROVE_OPTION) {
                            JarInfoView.setMessage(lblStatus, bundle.getString("JarInfo.extract.cancelled"),
                                    null, 10000);
                            return;  // Cancelled
                        }

                        File targetdir = chooser.getSelectedFile();
                        try {
                            if (!targetdir.exists()) targetdir.mkdir();
                            else {
                                String[] buttons = new String[]{bundle.getString("JarInfo.override.ok"),
                                        bundle.getString("JarInfo.override.cancel")};

                                // Ask user first if override is OK
                                retval = JOptionPane.showOptionDialog(
                                        JarInfoView.this, bundle.getString("JarInfo.override.msg"),
                                        bundle.getString("JarInfo.override.title"),
                                        JOptionPane.DEFAULT_OPTION, JOptionPane.WARNING_MESSAGE,
                                        null, buttons, buttons[1]);

                                if (retval != 0) {  // Cancel/close
                                    JarInfoView.setMessage(lblStatus, bundle.getString("JarInfo.extract.cancelled"),
                                            null, 10000);
                                    return;
                                }
                            }
                            if (targetdir.exists()) {
                                if (extractZipArchive(new File(jarFile), targetdir, bundle)) {
                                    JarInfoView.setMessage(lblStatus, bundle.getString("JarInfo.extract.success"),
                                            Boolean.TRUE, 10000);
                                } else {
                                    JarInfoView.setMessage(lblStatus, bundle.getString("JarInfo.extract.error"),
                                            Boolean.FALSE, 10000);
                                }
                            } else {
                                JarInfoView.setMessage(lblStatus, bundle.getString("JarInfo.extract.nodir"),
                                        Boolean.FALSE, 10000);
                            }
                        } catch (Exception e) {
                            JarInfoView.setMessage(lblStatus, bundle.getString("JarInfo.extract.error"),
                                    Boolean.FALSE, 10000);
                        }
                    }
                }, "JAR Info Extractor").start();
            }
        });

        // First of all display the notice that this is simply a JAR file
        boolean showTabNotice = this.jarInfo.props.getProperty("JAVAINFO_SHOW_TAB_NOTICE", "false").equals("true");
        if (showTabNotice) {
            String msg = this.jarInfo.replaceLinks(this.jarInfo.props.getProperty("JAVAINFO_NOTICE", ""));
            if (msg.trim().length() == 0) msg = bundle.getString("JarInfo.notice");
            StringBuffer builder = new StringBuffer(100);
            builder.append("<html><br>");
            builder.append("<table>");
            builder.append("<tr>");
            builder.append("<td>&nbsp;&nbsp;</td>");
            builder.append("<td>" + msg + "</td>");
            builder.append("</tr>");
            builder.append("</table>");
            builder.append("</html>");
            this.epNotice.setText(builder.toString());
        } else this.tabbedPane.remove(this.tabNotice);

        // Add the file name and the copyright notice if not deactivated
        boolean showFileName = this.jarInfo.props.getProperty("JAVAINFO_SHOW_FILENAME", "true").equals("true");
        boolean nameonly = this.jarInfo.props.getProperty("JAVAINFO_FILENAME_MODE", "fullpath").equals("nameonly");
        this.lblFileName.setVisible(showFileName);
        File fileref = new File(jarFile);
        this.lblFileName.setText("<html><font size='4'><i>" + (nameonly ? fileref.getName() : fileref.getAbsolutePath()) + "</i></font></html>");

        boolean showCopyright = this.jarInfo.props.getProperty("JAVAINFO_SHOW_COPYRIGHT", "true").equals("true");
        String copyright = this.jarInfo.props.getProperty("JAVAINFO_COPYRIGHT", "");
        if (copyright.trim().length() == 0) {
            copyright = "JarInfo &copy; 2005-2025 Leisenfels GmbH. All rights reserved.";
        }
        this.lblCopyright.setVisible(showCopyright);
        this.lblCopyright.setText("<html><i><font size='2'>" + copyright + "</font></i></html>");

        // Add the manifest contents
        boolean showTabManifest = this.jarInfo.props.getProperty("JAVAINFO_SHOW_TAB_MANIFEST", "auto").equals("true");
        boolean showTabAuto = this.jarInfo.props.getProperty("JAVAINFO_SHOW_TAB_MANIFEST", "auto").equals("auto");

        String manifest = null;
        if (this.jarInfo.manifestURL != null) { // From classpath
            manifest = JarInfo.loadEntry(this.jarInfo.manifestURL);
        } else { // From external JAR
            try {
                manifest = JarInfo.loadEntry(this.jarInfo.jarFile, "META-INF/MANIFEST.MF");
            } catch (Exception ignored) {
            }
        }
        if (showTabManifest || (showTabAuto && manifest != null)) {
            boolean unwrapped = this.jarInfo.props.getProperty("JAVAINFO_MANIFEST_MODE", "plain_unwrapped").equals("plain_unwrapped");
            if (unwrapped) {  // Concatenate wrapped lines starting with a SPACE
                // "Implementation-URL: http://leisenfels.al11.net/Wiki.jsp?page=ProjectCo"
                // " mmons"
                // --> "Implementation-URL: http://leisenfels.al11.net/Wiki.jsp?page=ProjectCommons"
                manifest = manifest.replaceAll("(([\r][\n]|[\n]|[\r])[ ])", "");
            }
            this.taManifest.setText(manifest);
        } else this.tabbedPane.remove(this.tabManifest);

        Vector services = new Vector(0);  // Java 1.4!

        // Add the entries list, determine service providers on-the-fly
        boolean showTabEntries = this.jarInfo.props.getProperty("JAVAINFO_SHOW_TAB_ENTRIES", "true").equals("true");
        if (showTabEntries) {

            Vector entries = JarInfoView.searchFilesInZIP(jarFile); // Tree mode only
            int size = entries.size();
            if (size > 0) {

                // Convert ZipEntry container to String container to allow sorting
                final Vector sorter = new Vector(0);  // Java 1.4!
                for (int i = 0; i < size; i++) {
                    next = (ZipEntry) entries.elementAt(i);
                    sorter.addElement(next.getName());  // Java 1.4!

                    // Obfuscated/minimized JARs may not contain any directory entries (required for tree)
                    if (!next.isDirectory()) {
                        Vector parents = JarInfoView.generateParentDirectories(next);
                        int psize = parents.size();
                        for (int p = 0; p < psize; p++) {
                            String parent = (String) parents.elementAt(p);
                            if (!sorter.contains(parent)) sorter.addElement(parent);  // Java 1.4!
                            // Sorting below sets proper order parent before child
                        }
                    }

                    // Extract the services displayed separately
                    if (next.getName().startsWith("META-INF/services/") && !next.isDirectory() &&
                            next.getName().substring(18).indexOf('/') == -1) {
                        services.addElement(next.getName().substring(18));  // Java 1.4!
                    }
                }
                Collections.sort(sorter);  // Java 1.4!

                boolean textmode = this.jarInfo.props.getProperty("JAVAINFO_ENTRIES_MODE", "text").equals("text");
                if (textmode) {
                    ((CardLayout) this.tabEntries.getLayout()).show(this.tabEntries, "text");

                    StringBuffer builder = new StringBuffer(0);
                    for (int i = 0; i < size; i++) {
                        builder.append(sorter.elementAt(i));
                        builder.append('\n');
                    }
                    this.textAreaEntries.setText(builder.toString());
                } else {  // Use the tree mode
                    ((CardLayout) this.tabEntries.getLayout()).show(this.tabEntries, "tree");

                    this.treeEntries.setShowsRootHandles(true);
                    this.rootNode = new DefaultMutableTreeNode();
                    this.treeModel = new DefaultTreeModel(this.rootNode);
                    this.treeEntries.setRootVisible(false);
                    this.treeEntries.setModel(this.treeModel);

                    SwingUtilities.invokeLater(new Runnable() {  // Requires EDT!

                        public void run() {
                            boolean expand = jarInfo.props.getProperty("JAVAINFO_ENTRIES_EXPAND_TREE", "false").equals("true");
                            populateTree(rootNode, sorter, expand);  // Add nodes
                        }
                    });
                }
                entries.removeAllElements();
                // sorter must not be cleared here!
            }
            boolean showExtract = this.jarInfo.props.getProperty("JAVAINFO_ENTRIES_SHOW_EXTRACT", "false").equals("true");
            this.btnExtract.setVisible(showExtract);
        } else this.tabbedPane.remove(this.tabEntries);

        // Any service providers present?
        boolean showTabServices = this.jarInfo.props.getProperty("JAVAINFO_SHOW_TAB_SERVICES", "auto").equals("true");
        showTabAuto = this.jarInfo.props.getProperty("JAVAINFO_SHOW_TAB_SERVICES", "auto").equals("auto");
        if (showTabServices || (showTabAuto && !services.isEmpty())) {
            this.listServices.setListData(services);
            this.listServices.addListSelectionListener(this);

            // Auto-select a service?
            String autoselect = this.jarInfo.props.getProperty("JAVAINFO_SERVICES_AUTO_SELECT", "");
            //if (autoselect.trim().length() == 0 && services.size() == 1) {  // Select first and only
            //autoselect = (String) services.elementAt(0);
            //}
            if (autoselect.trim().length() >= 1) {
                int size = services.size();
                for (int i = 0; i < size; i++) {
                    String nextitem = (String) services.elementAt(i);
                    if (nextitem.equals(autoselect)) {
                        this.listServices.setSelectedIndex(i);
                        break;
                    }
                }
            }
        } else this.tabbedPane.remove(this.tabServices);  // Do not show the tab at all

        this.pack();  // Necessary for Linux!

        Toolkit toolkit = this.getToolkit();
        Dimension dims = toolkit.getScreenSize();

        // Place the window in the center of the screen or use user settings
        int defaultWidth = -1;
        try {
            defaultWidth = Integer.parseInt(this.jarInfo.props.getProperty("JAVAINFO_WINDOW_WIDTH"));
        } catch (Exception ignored) {
        }
        if (defaultWidth <= 0) defaultWidth = 580;

        int defaultHeight = -1;
        try {
            defaultHeight = Integer.parseInt(this.jarInfo.props.getProperty("JAVAINFO_WINDOW_HEIGHT"));
        } catch (Exception ignored) {
        }
        if (defaultHeight <= 0) defaultHeight = 420;

        int defaultX = -1;  // Negative = center
        try {
            defaultX = Integer.parseInt(this.jarInfo.props.getProperty("JAVAINFO_WINDOW_LOCATION_X"));
        } catch (Exception ignored) {
        }
        if (defaultX < 0) defaultX = (int) ((dims.width - defaultWidth) / 2.0);

        int defaultY = -1;  // Negative = center
        try {
            defaultY = Integer.parseInt(this.jarInfo.props.getProperty("JAVAINFO_WINDOW_LOCATION_Y"));
        } catch (Exception ignored) {
        }
        if (defaultY < 0) defaultY = (int) ((dims.height - defaultHeight) / 2.0) - 10;

        this.setSize(defaultWidth, defaultHeight);
        this.setLocation(defaultX, defaultY);
        this.splitPaneServices.setDividerLocation(130);

        try {
            // Set initial tab selection
            this.tabbedPane.setSelectedIndex(0);
            String value = this.jarInfo.props.getProperty("JAVAINFO_INITIAL_TAB", "");
            if (value.equals("notice")) this.tabbedPane.setSelectedComponent(this.tabNotice);
            else if (value.equals("manifest")) {
                this.tabbedPane.setSelectedComponent(this.tabManifest);
            } else if (value.equals("entries")) this.tabbedPane.setSelectedComponent(this.tabEntries);
            else if (value.equals("services")) {
                this.tabbedPane.setSelectedComponent(this.tabServices);
            }
        } catch (Exception ignored) {
        }

        // Finally setup the look & feel
        String plaf = this.jarInfo.props.getProperty("JAVAINFO_PLAF", "auto");
        if (!plaf.equals("auto")) {  // Use platform default otherwise
            try {
                UIManager.setLookAndFeel(plaf);
                SwingUtilities.updateComponentTreeUI(this);
            } catch (Exception ignored) {
            }
        }
        this.setVisible(true);
    }

    /**
     * Searches the given JAR file (maybe a ZIP file as well) for entries.
     *
     * @param zipFile The JAR/ZIP file to search for files
     * @return The unsorted container with entries
     */
    private static Vector searchFilesInZIP(String zipFile) {

        Vector result = new Vector(0);

        try {
            ZipFile zip = new ZipFile(zipFile);
            Enumeration enumeration = zip.entries();
            while (enumeration.hasMoreElements()) {
                ZipEntry entry = (ZipEntry) enumeration.nextElement();
                result.addElement(entry);  // Java 1.4!
            }
        } catch (Exception ignored) {
        }
        return result;
    }

    /**
     * Generates all parent directory paths for the given (file) ZIP entry.
     *
     * @param entry The JAR/ZIP file to search for files
     * @return The container with entries in alp
     */
    private static Vector generateParentDirectories(ZipEntry entry) {

        Vector parents = new Vector(0);

        if (entry.isDirectory()) return parents;

        try {
            String path = entry.getName();
            // e.g. com/lf/commons/info/resource/jarinfo.properties
            String[] parts = path.split("[/]");
            for (int i = 0; i < parts.length - 1; i++) {
                String parent = concatArray(parts, i, "/") + "/";
                parents.add(parent);  // Java 1.4!
            }
        } catch (Exception ignored) {
        }
        return parents;
    }

    /**
     * Simply concatenates an array of <code>String</code>'s and adds separators if necessary.
     *
     * @param parts     The parts of the new character sequence
     * @param lastindex Stop at this index (0-based, included)
     * @param separator String used as separator for the concatenation
     * @return Concatenation result
     * @throws ArrayIndexOutOfBoundsException If <code>index</code> is invalid
     */
    private static String concatArray(String[] parts, int lastindex, String separator) {

        StringBuffer result = new StringBuffer(0);

        for (int i = 0; i <= lastindex; i++) {
            if (result.length() > 0) result.append(separator);
            result.append(parts[i]);
        }
        return result.toString();
    }

    /**
     * Creates the nodes of the entries tree.
     *
     * @param rootnode The parental node
     * @param sorter   The sorted entries
     * @param expand   Expand all nodes?
     */
    private void populateTree(DefaultMutableTreeNode rootnode, Vector sorter, boolean expand) {

        String entry;
        Hashtable folders = new Hashtable(0);
        DefaultMutableTreeNode node, parent;
        File file;
        TreePath path;

        try {
            int count = sorter.size();
            for (int i = 0; i < count; i++) {
                entry = '/' + (String) sorter.elementAt(i);  // e.g. "/META-INF/MANIFEST.MF"
                file = new File(entry);

                // Get parent node
                node = new DefaultMutableTreeNode(file.getName());
                parent = (DefaultMutableTreeNode) folders.get(file.getParentFile());
                if (parent == null) parent = rootnode;
                parent.add(node);
                if (entry.endsWith("/")) folders.put(file, node);
            }

            // Expand root node to make it become visible
            path = new TreePath(this.rootNode.getPath());
            this.treeEntries.expandPath(path);
            this.treeEntries.makeVisible(path);

            if (expand) {
                Enumeration en = folders.keys();
                while (en.hasMoreElements()) {
                    node = (DefaultMutableTreeNode) folders.get(en.nextElement());
                    path = new TreePath(node.getPath());
                    this.treeEntries.expandPath(path);
                    this.treeEntries.makeVisible(path);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            folders.clear();
            sorter.removeAllElements();
        }
    }

    /**
     * Calculates the position for a dialog window on the screen.
     * <p/>
     * Can be used to center a dialog for its parent window. If the parent window is <code>null</code>
     * then the point is calculated for the whole desktop automatically.
     *
     * @param parent The parent window to center the dialog, for desktop if <code>null</code>
     * @param width  The width of the dialog to be centered
     * @param height The height of the dialog to be centered
     * @return The position on the screen
     */
    public static Point centerForWindow(Window parent, int width, int height) {

        // Calculate the parent center, use desktop if parent is not available
        Toolkit toolkit = Toolkit.getDefaultToolkit();
        Dimension size = parent != null ? parent.getSize() : toolkit.getScreenSize();
        Point pos = parent != null ? parent.getLocation() : new Point(0, 0);
        int centerX = pos.x + size.width / 2;
        int centerY = pos.y + size.height / 2;

        // Now calculate the dialog upper left corner
        int locationX = Math.max(0, centerX - width / 2);
        int locationY = Math.max(0, centerY - height / 2);

        return new Point(locationX, locationY);
    }

    /**
     * This extracts a given local ZIP/JAR file directly to a specified directory.
     * <p/>
     * The archive is first of all copied to the user's home directory since we cannot extract
     * non-local files (<code>ZipFile</code> has only constructor with <code>File</code> parameter).
     * This method uses the name of the ZIP archive directly. Then the file is extracted.
     *
     * @param zipfile   ZIP/JAR archive to extract
     * @param targetdir The local destination directory
     * @param bundle    The strings
     * @return <code>true</code>, if saving worked, <code>false</code> else
     */
    public boolean extractZipArchive(File zipfile, File targetdir, ResourceBundle bundle) {

        ZipEntry next;
        File newfile;
        boolean success = false;

        if (zipfile == null || targetdir == null) return false;
        this.btnExtract.setEnabled(false);

        try {
            ZipFile zip = new ZipFile(zipfile);

            Enumeration enumeration = zip.entries();
            while (enumeration.hasMoreElements()) {
                next = (ZipEntry) enumeration.nextElement();
                newfile = new File(targetdir + File.separator + next.getName());
                if (next.isDirectory()) {  // Create directory
                    if (!newfile.exists()) {
                        if (!newfile.mkdir()) throw new Exception();
                    }
                } else {  // Extract ZIP entry, create file

                    // Obfuscated/minimized JARs may not contain any directory entries (required for tree)
                    Vector parents = JarInfoView.generateParentDirectories(next);
                    int psize = parents.size();
                    for (int p = 0; p < psize; p++) {
                        String parent = (String) parents.elementAt(p);
                        File parentfile = new File(targetdir + File.separator + parent);
                        if (!parentfile.exists()) {
                            if (!parentfile.mkdir()) throw new Exception();
                        }
                    }

                    if (!saveZIPEntry(zip, next, newfile)) throw new Exception();
                    JarInfoView.setMessage(lblStatus,
                            bundle.getString("JarInfo.extract.entry") + ' ' + next.getName(),
                            null, -1);
                }
            }
            zip.close();
            success = true;
        } catch (Exception ignored) {
        } finally {
            this.btnExtract.setEnabled(true);
        }
        return success;
    }

    /**
     * This method stores the content of a ZIP/JAR entry to a local file.
     *
     * @param archive ZIP/JAR archive to copy file from
     * @param entry   Source to be copied
     * @param target  The local file to store the content
     * @return <code>true</code>, if saving worked
     */
    public static boolean saveZIPEntry(ZipFile archive, ZipEntry entry, File target) {

        boolean error = false;
        InputStream in = null;
        FileOutputStream out = null;
        byte[] buffer = new byte[1024];
        int count;

        // After checking the validity of the source URL
        // we simply start copying the content.

        if (archive != null && entry != null && target != null) {
            try {
                in = archive.getInputStream(entry);
            } catch (Exception e2) {
                error = true;
            }
            if (!error) {

                // We have the input stream now.
                // Let's write the data to the
                // local destination file.

                try {
                    out = new FileOutputStream(target);
                } catch (Exception e3) {
                    error = true;
                }
                if (!error) {

                    // We copy the data now
                    count = 1;
                    while (count > 0 && !error) {
                        try {
                            count = in.read(buffer);
                        } catch (Exception e4) {
                            error = true;
                        }
                        if (!error && count > 0) {
                            try {
                                out.write(buffer, 0, count);
                            } catch (Exception e5) {
                                error = true;
                            }
                        }
                    }

                    // We close the streams and in the error
                    // case we additionally try to remove the
                    // output file (artifacts).
                    try {
                        in.close();
                    } catch (Exception ignored) {
                    }
                    try {
                        out.close();
                    } catch (Exception ignored) {
                    }

                    if (error) {
                        try {
                            target.delete();
                        } catch (Exception ignored) {
                        }
                    }
                }
            }
        }
        return !error;
    }

    /**
     * Prints a message for the user.
     *
     * @param label   The component
     * @param text    The text
     * @param success Successful? <code>null</code>=default font color
     * @param millis  Shown for n milliseconds, nonstop if -1
     */
    private static void setMessage(final JLabel label, String text, Boolean success, final int millis) {

        if (success == null) label.setForeground(UIManager.getColor("Label.foreground"));
        else if (success.booleanValue()) label.setForeground(Color.green.darker());
        else label.setForeground(Color.red);

        label.setText(text);

        if (millis >= 0) {
            new Thread(new Runnable() {
                public void run() {
                    try {
                        Thread.sleep(millis);
                        label.setText("");
                    } catch (Exception ignored) {
                    }
                }
            }).start();
        }
    }

    private void initComponents() {
        // JFormDesigner - Component initialization - DO NOT MODIFY  //GEN-BEGIN:initComponents
        this.pnlMain = new JPanel();
        JPanel panel2 = new JPanel();
        JPanel panel3 = new JPanel();
        this.lblCopyright = new JLabel();
        JPanel panel1 = new JPanel();
        this.lblStatus = new JLabel();
        JPanel panel4 = new JPanel();
        this.btnExtract = new JButton();
        this.btnClose = new JButton();
        this.tabbedPane = new JTabbedPane();
        this.tabNotice = new JPanel();
        JScrollPane scrNotice = new JScrollPane();
        this.epNotice = new JEditorPane();
        this.tabManifest = new JPanel();
        JScrollPane scrManifest = new JScrollPane();
        this.taManifest = new JTextArea();
        this.tabEntries = new JPanel();
        this.scrEntriesText = new JScrollPane();
        this.textAreaEntries = new JTextArea();
        this.scrEntriesTree = new JScrollPane();
        this.treeEntries = new JTree();
        this.tabServices = new JPanel();
        this.splitPaneServices = new JSplitPane();
        JScrollPane scrollPane1 = new JScrollPane();
        this.listServices = new JList();
        JScrollPane scrollPane2 = new JScrollPane();
        this.textAreaProviders = new JTextArea();
        JPanel panel5 = new JPanel();
        this.lblFileName = new JLabel();

        //======== pnlMain ========
        {
            this.pnlMain.setLayout(new BorderLayout(5, 5));

            //======== panel2 ========
            {
                panel2.setBorder(new EmptyBorder(0, 5, 5, 0));
                panel2.setLayout(new BorderLayout());

                //======== panel3 ========
                {
                    panel3.setLayout(new BorderLayout());

                    //---- lblCopyright ----
                    this.lblCopyright.setText("Copyright");
                    this.lblCopyright.setVerticalAlignment(SwingConstants.TOP);
                    panel3.add(this.lblCopyright, BorderLayout.NORTH);

                    //======== panel1 ========
                    {
                        panel1.setLayout(new BorderLayout());

                        //---- lblStatus ----
                        this.lblStatus.setText("Status");
                        panel1.add(this.lblStatus, BorderLayout.CENTER);

                        //======== panel4 ========
                        {
                            panel4.setLayout(new FlowLayout());

                            //---- btnExtract ----
                            this.btnExtract.setText("Extract");
                            this.btnExtract.setVerticalAlignment(SwingConstants.BOTTOM);
                            panel4.add(this.btnExtract);

                            //---- btnClose ----
                            this.btnClose.setText("Close");
                            this.btnClose.setVerticalAlignment(SwingConstants.BOTTOM);
                            panel4.add(this.btnClose);
                        }
                        panel1.add(panel4, BorderLayout.EAST);
                    }
                    panel3.add(panel1, BorderLayout.SOUTH);
                }
                panel2.add(panel3, BorderLayout.CENTER);
            }
            this.pnlMain.add(panel2, BorderLayout.SOUTH);

            //======== tabbedPane ========
            {
                this.tabbedPane.setBorder(new EmptyBorder(0, 5, 0, 5));

                //======== tabNotice ========
                {
                    this.tabNotice.setLayout(new BorderLayout());

                    //======== scrNotice ========
                    {

                        //---- epNotice ----
                        this.epNotice.setEnabled(true);
                        this.epNotice.setContentType("text/html");
                        this.epNotice.setEditable(false);
                        scrNotice.setViewportView(this.epNotice);
                    }
                    this.tabNotice.add(scrNotice, BorderLayout.CENTER);
                }
                this.tabbedPane.addTab("Notice", this.tabNotice);


                //======== tabManifest ========
                {
                    this.tabManifest.setLayout(new BorderLayout());

                    //======== scrManifest ========
                    {
                        scrManifest.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
                        scrManifest.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);

                        //---- taManifest ----
                        this.taManifest.setEditable(false);
                        this.taManifest.setFont(new Font("Monospaced", Font.PLAIN, 12));
                        scrManifest.setViewportView(this.taManifest);
                    }
                    this.tabManifest.add(scrManifest, BorderLayout.CENTER);
                }
                this.tabbedPane.addTab("Manifest", this.tabManifest);


                //======== tabEntries ========
                {
                    this.tabEntries.setLayout(new CardLayout());

                    //======== scrEntriesText ========
                    {

                        //---- textAreaEntries ----
                        this.textAreaEntries.setEditable(false);
                        this.textAreaEntries.setFont(new Font("Monospaced", Font.PLAIN, 12));
                        this.scrEntriesText.setViewportView(this.textAreaEntries);
                    }
                    this.tabEntries.add(this.scrEntriesText, "text");

                    //======== scrEntriesTree ========
                    {

                        //---- treeEntries ----
                        this.treeEntries.setEditable(false);
                        this.treeEntries.setLargeModel(true);
                        this.scrEntriesTree.setViewportView(this.treeEntries);
                    }
                    this.tabEntries.add(this.scrEntriesTree, "tree");
                }
                this.tabbedPane.addTab("Entries", this.tabEntries);


                //======== tabServices ========
                {
                    this.tabServices.setLayout(new BorderLayout());

                    //======== splitPaneServices ========
                    {
                        this.splitPaneServices.setOrientation(JSplitPane.VERTICAL_SPLIT);

                        //======== scrollPane1 ========
                        {

                            //---- listServices ----
                            this.listServices.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
                            scrollPane1.setViewportView(this.listServices);
                        }
                        this.splitPaneServices.setTopComponent(scrollPane1);

                        //======== scrollPane2 ========
                        {

                            //---- textAreaProviders ----
                            this.textAreaProviders.setEditable(false);
                            this.textAreaProviders.setFont(new Font("Monospaced", Font.PLAIN, 12));
                            scrollPane2.setViewportView(this.textAreaProviders);
                        }
                        this.splitPaneServices.setBottomComponent(scrollPane2);
                    }
                    this.tabServices.add(this.splitPaneServices, BorderLayout.CENTER);
                }
                this.tabbedPane.addTab("Services", this.tabServices);

            }
            this.pnlMain.add(this.tabbedPane, BorderLayout.CENTER);

            //======== panel5 ========
            {
                panel5.setBorder(new EmptyBorder(10, 10, 5, 10));
                panel5.setLayout(new FlowLayout(FlowLayout.LEFT, 0, 0));

                //---- lblFileName ----
                this.lblFileName.setText("commons-1.6.1.jar");
                panel5.add(this.lblFileName);
            }
            this.pnlMain.add(panel5, BorderLayout.NORTH);
        }
        // JFormDesigner - End of component initialization  //GEN-END:initComponents
    }

    // JFormDesigner - Variables declaration - DO NOT MODIFY  //GEN-BEGIN:variables
    private JPanel pnlMain;
    private JLabel lblCopyright;
    private JLabel lblStatus;
    private JButton btnExtract;
    private JButton btnClose;
    private JTabbedPane tabbedPane;
    private JPanel tabNotice;
    private JEditorPane epNotice;
    private JPanel tabManifest;
    private JTextArea taManifest;
    private JPanel tabEntries;
    private JScrollPane scrEntriesText;
    private JTextArea textAreaEntries;
    private JScrollPane scrEntriesTree;
    private JTree treeEntries;
    private JPanel tabServices;
    private JSplitPane splitPaneServices;
    private JList listServices;
    private JTextArea textAreaProviders;
    private JLabel lblFileName;
    // JFormDesigner - End of variables declaration  //GEN-END:variables

    /**
     * Called whenever the value of the selection changes.
     *
     * @param e Event object reference
     */
    public void valueChanged(ListSelectionEvent e) {

        this.textAreaProviders.setText("");

        String name = (String) this.listServices.getSelectedValue();
        if (name == null) return;

        try {
            // Derive target URL from manifest URL (see above notice)
            String temp = this.jarInfo.manifestURL.toString();
            URL target = new URL(temp.replaceFirst("(MANIFEST.MF$)", "services/" + name));

            String contents = JarInfo.loadEntry(target);
            if (contents != null) this.textAreaProviders.setText(contents);
        } catch (Exception ignored) {
        }
    }
}
