/*
    Copyright (c) 2005-2025 Leisenfels GmbH GmbH GmbH. All rights reserved.
    Use is subject to license terms.
*/

package com.lf.commons.info;

import java.awt.*;
import java.io.*;
import java.net.URL;
import java.net.URLConnection;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;


/**
 * Lightweight main class presented when the module JAR is executed directly.
 * <p/>
 * If the client computer has a graphical output screen then a simple dialog frame is presented.
 * If the client lacks GUI support, the infos are presented over the command line interface.
 * Both variants present the contents of the MANIFEST.MF file. Remember that this class must not
 * use any Java classes either included in the JRE or that are part of this JAR file. For example
 * the VFS classes are not accessible and must therefore not be used from this class. The easiest
 * way to deal with is to simply utilize the standard Java JRE classes. So this class can be
 * integrated into any standalone package.
 *
 * @author Axel Schwolow
 * @created 2016-01-01
 * @since 1.4
 */
public class JarInfo {


    /**
     * Main graphical component.
     */
    protected JarInfoView view;
    /**
     * URL for accessing the JAR manifest file from classpath, otherwise <code>jarFile</code> is used.
     */
    protected URL manifestURL = null;
    /**
     * Entity to access the JAR manifest from an external JAR file, otherwise <code>manifestURL</code> is used.
     */
    protected ZipFile jarFile = null;
    /**
     * Values from 'jarinfo.properties' file (optional).
     */
    protected Properties props;


    /**
     * Constructor method for the manifest display.
     *
     * @param cliJARFile Optional JAR file to process, classpath is analyzed otherwise
     * @param cliProps   Optional configuration how to display JAR contents
     * @param headless   Optional param to run application in headless mode
     */
    public JarInfo(String cliJARFile, String cliProps, Boolean headless) {

        if ((headless != null && !headless.booleanValue()) || !GraphicsEnvironment.isHeadless()) {
            this.view = new JarInfoView(this);
        }

        // Set the default values as documented
        this.props = new Properties();
        this.props.setProperty("JAVAINFO_LOCALE", "auto");
        this.props.setProperty("JAVAINFO_PLAF", "auto");
        this.props.setProperty("JAVAINFO_WINDOW_TITLE", "JarInfo");
        this.props.setProperty("JAVAINFO_WINDOW_ICON", "");
        this.props.setProperty("JAVAINFO_WINDOW_WIDTH", "580");
        this.props.setProperty("JAVAINFO_WINDOW_HEIGHT", "420");
        this.props.setProperty("JAVAINFO_WINDOW_LOCATION_X", "center");
        this.props.setProperty("JAVAINFO_WINDOW_LOCATION_Y", "center");
        this.props.setProperty("JAVAINFO_INITIAL_TAB", "");
        this.props.setProperty("JAVAINFO_SHOW_FILENAME", "true");
        this.props.setProperty("JAVAINFO_FILENAME_MODE", "fullpath");
        this.props.setProperty("JAVAINFO_SHOW_COPYRIGHT", "true");
        this.props.setProperty("JAVAINFO_COPYRIGHT", "");
        this.props.setProperty("JAVAINFO_SHOW_TAB_NOTICE", "false");
        this.props.setProperty("JAVAINFO_NOTICE", "");
        this.props.setProperty("JAVAINFO_SHOW_TAB_MANIFEST", "true");
        this.props.setProperty("JAVAINFO_MANIFEST_MODE", "plain_unwrapped");
        this.props.setProperty("JAVAINFO_SHOW_TAB_ENTRIES", "true");
        this.props.setProperty("JAVAINFO_ENTRIES_MODE", "text");
        this.props.setProperty("JAVAINFO_ENTRIES_EXPAND_TREE", "false");
        this.props.setProperty("JAVAINFO_ENTRIES_SHOW_EXTRACT", "false");
        this.props.setProperty("JAVAINFO_SHOW_TAB_SERVICES", "auto");
        this.props.setProperty("JAVAINFO_SERVICES_AUTO_SELECT", "");

        Properties temp = new Properties();
        try {
            if (cliProps != null) {
                temp.load(new FileInputStream(cliProps)); // From CLI param
                if (!temp.isEmpty()) {
                    Enumeration en = temp.keys();  // Java 1.4
                    while (en.hasMoreElements()) {
                        String key = (String) en.nextElement();
                        this.props.setProperty(key, temp.getProperty(key));
                    }
                }
            } else {
                // Get the default values from 'jarinfo.properties' in classpath
                temp.load(JarInfo.class.getResourceAsStream("resource/jarinfo.properties"));
                if (!temp.isEmpty()) {
                    Enumeration en = temp.keys();  // Java 1.4
                    while (en.hasMoreElements()) {
                        String key = (String) en.nextElement();
                        this.props.setProperty(key, temp.getProperty(key));
                    }
                }
            }
        } catch (Exception ignored) {
        } finally {
            temp.clear();
        }

        String jarFilePath = null;
        try {
            if (cliJARFile != null) { // From CLI param
                this.jarFile = new ZipFile(cliJARFile);
                jarFilePath = cliJARFile;
            } else {
                // Analyze classpath

                // Caution: Some JRE (e.g. Linux) will return the MANIFEST-MF file for rt.jar
                // instead of the MANIFEST.MF from our JAR archive! We have to find one of the
                // resource files and create the access path for the manifest file manually.
                String clazz = JarInfo.class.getName();
                String dummy = clazz.substring(clazz.lastIndexOf('.') + 1) + ".properties";  // English default
                String target = JarInfo.class.getResource(dummy).toString();
                String packg = JarInfo.class.getPackage().getName().replaceAll("([\\.])", "/");
                this.manifestURL = new URL(target.replaceFirst('(' + packg + '/' + dummy + ')',
                        "META-INF/MANIFEST.MF"));

                jarFilePath = target.replaceFirst("([\\!]/" + packg + '/' + dummy + ')', "");
                if (jarFilePath.startsWith("jar:file:")) {
                    jarFilePath = jarFilePath.replaceFirst("(^jar:file:)", "");
                    jarFilePath = new File(jarFilePath).toString();  // Normalize path (Windows)
                }
            }
        } catch (Exception ignored) {
        }

        String language = this.props.getProperty("JAVAINFO_LOCALE", "auto");
        if (!language.equals("auto")) {
            try {
                // Check out value and set system property accordingly.
                // We cannot use the ISO 639-2 codes due to Java 1.4 compatibility.
                if (language.equals("en")) Locale.setDefault(Locale.ENGLISH);
                else if (language.equals("de")) {
                    Locale.setDefault(Locale.GERMAN);
                }
                // Default is the system default for the user, not "user.language"!
            } catch (Exception ignored) {
            }
        }
        ResourceBundle bundle = ResourceBundle.getBundle(JarInfo.class.getName());

        // Mode 1: Provide information via CLI since the JVM is headless (e.g. server)
        if ((headless != null && headless.booleanValue()) || GraphicsEnvironment.isHeadless()) {
            printInfo(jarFilePath, bundle);
            try {
                Thread.sleep(500);  // Give IDE some time to flush all messages
            } catch (Exception ignored) {
            }
        } else {
            // Mode 2: Client supports GUI so let's display the dialog
            this.view.showWindow(jarFilePath, bundle);
        }
    }

    /**
     * Prints the information from MANIFEST.MF via <code>System.out</code>.
     *
     * @param jarFile The path of the JAR file, maybe <code>null</code>
     * @param bundle  The string resources
     */
    private void printInfo(String jarFile, ResourceBundle bundle) {

        boolean nameonly = this.props.getProperty("JAVAINFO_FILENAME_MODE", "fullpath").equals("nameonly");
        File fileref = new File(jarFile);

        String msg;
        boolean showTabNotice = this.props.getProperty("JAVAINFO_SHOW_TAB_NOTICE", "false").equals("true");
        if (showTabNotice) {
            msg = this.replaceLinks(this.props.getProperty("JAVAINFO_NOTICE", ""));
            if (msg.trim().length() == 0) msg = bundle.getString("JarInfo.notice");
        }

        // Give the user a hint about the type of file
        System.out.println();
        System.out.println(nameonly ? fileref.getName() : fileref.getAbsolutePath());
        System.out.println();

        // Let's print the original MANIFEST.MF file via System.out print stream.
        // Don't use Manifest.write() since the attribute order is not guaranteed.
        boolean showTabManifest = this.props.getProperty("JAVAINFO_SHOW_TAB_MANIFEST", "auto").equals("true");
        boolean showTabAuto = this.props.getProperty("JAVAINFO_SHOW_TAB_MANIFEST", "auto").equals("auto");

        String manifest = null;
        if (this.manifestURL != null) { // From classpath
            manifest = JarInfo.loadEntry(this.manifestURL);
        } else { // From external JAR
            try {
                manifest = JarInfo.loadEntry(this.jarFile, "META-INF/MANIFEST.MF");
            } catch (Exception ignored) {
            }
        }
        if (showTabManifest || (showTabAuto && manifest != null)) {
            boolean unwrapped = this.props.getProperty("JAVAINFO_MANIFEST_MODE", "plain_unwrapped").equals("plain_unwrapped");
            if (unwrapped) {  // Concatenate wrapped lines starting with a SPACE
                // "Implementation-URL: http://leisenfels.al11.net/Wiki.jsp?page=ProjectCo"
                // " mmons"
                // --> "Implementation-URL: http://leisenfels.al11.net/Wiki.jsp?page=ProjectCommons"
                manifest = manifest.replaceAll("(([\r][\n]|[\n]|[\r])[ ])", "");
            }
            System.out.println(manifest);
        }
    }

    /**
     * Loads file data from the JAR archive (e.g. manifest).
     *
     * @param target The target URL for the JAR file
     * @return File data, maybe <code>null</code>
     */
    protected static String loadEntry(URL target) {

        try {
            // Let's print the original MANIFEST.MF file via System.out print stream.
            // Don't use Manifest.write() since the attribute order is not guaranteed.
            URLConnection connect = target.openConnection();
            connect.connect();

            String newline = System.getProperty("line.separator");
            StringBuffer buffer = new StringBuffer(0);  // Java 1.4!
            String line;

            InputStreamReader isreader = new InputStreamReader(target.openStream(), "ISO-8859-1");
            LineNumberReader reader = new LineNumberReader(isreader);
            while ((line = reader.readLine()) != null) {
                buffer.append(line);
                buffer.append(newline);
            }
            reader.close();

            return buffer.toString();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Loads file data from the JAR archive (e.g. manifest).
     *
     * @param jarFile The external JAR file
     * @param entry   The path for the entry to read
     * @return File data, maybe <code>null</code>
     */
    protected static String loadEntry(ZipFile jarFile, String entry) {

        InputStream stream;
        byte[] buffer = new byte[1024];

        try {
            // Let's print the original MANIFEST.MF file via System.out print stream.
            // Don't use Manifest.write() since the attribute order is not guaranteed.
            ZipEntry zipEntry = jarFile.getEntry(entry);

            // Get the byte stream
            byte[] result = new byte[0];
            int bytes = 0;
            stream = jarFile.getInputStream(zipEntry);

            while (bytes != -1) {
                bytes = stream.read(buffer);
                if (bytes != -1) {
                    byte[] temp = new byte[result.length + bytes];
                    System.arraycopy(result, 0, temp, 0, result.length);
                    System.arraycopy(buffer, 0, temp, result.length, bytes);
                    result = temp;
                }
            }
            stream.close();
            return new String(result, "ISO-8859-1");
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Replaces resources as part of HTML notices texts by proper links into the JAR classpath.
     *
     * @param text The raw text
     * @return The replaced text
     */
    protected String replaceLinks(String text) {

        URL target;
        String pattern;
        StringBuffer buffer = new StringBuffer(0);
        buffer.append(text);

        try {
            Vector fragments = JarInfo.extractAll(text, "(%[^%]+%)");  // e.g. "%information_32.png%"
            int size = fragments.size();
            for (int i = 0; i < size; i++) {
                pattern = (String) fragments.elementAt(i);
                try {
                    target = this.getClass().getResource("resource/" + pattern.replaceAll("(^%|%$)", ""));
                    JarInfo.replace(buffer, pattern, target.toString());
                    // e.g. "jar:file:/L:/lf/commons/dist/base/commons-1.6.1/lib/commons-eng_US-1.6.1.jar!/com/lf/commons/info/resource/information_32.png"
                } catch (Exception ignored) {
                }
            }
        } catch (Exception ignored) {
        }
        return buffer.toString();
    }

    /**
     * Searches strings for certain patterns and returns all results.
     * <p/>
     * Search patterns are defined in shape of regular expressions used by
     * the method "String.matches" (since SDK 1.4.1). See the Java API
     * documentation for details.
     *
     * @param string The string to be searched
     * @param regex  The regular expression pattern to be searched for
     * @return List of substrings, may be empty
     */
    private static Vector extractAll(String string, String regex) {

        Vector result = new Vector(0);

        try {
            Pattern pattern = Pattern.compile(regex);
            Matcher matcher = pattern.matcher(string);
            while (matcher.find()) result.addElement(matcher.group());
        } catch (Exception ignored) {
        }
        return result;
    }

    /**
     * Replaces a specific pattern (sub-string) in a <code>String</code>.
     * <p/>
     * The return value counts the replacements. If the <code>substitute</code>
     * parameter is <code>null</code> the found pattern substrings are simply
     * removed from string. Since this method uses the fast searching routines
     * by Knuth-Morris-Pratt the searched <code>String</code> objects may be
     * large.
     *
     * @param buffer     The <code>StringBuffer</code> object for search & replace
     * @param pattern    This sub-string is the search target
     * @param substitute The found patterns are substituted by this
     */
    private static void replace(StringBuffer buffer, String pattern, String substitute) {

        int count = 0;
        boolean finished = false;
        int index;


        if (buffer != null && pattern != null) {
            // We try as long as the searching method completes
            while (!finished) {

                index = JarInfo.searchIndex(buffer.toString(), pattern);
                if (index != -1) {

                    // Method found something!
                    // We have to replace the pattern by the substitute.
                    buffer.replace(index - pattern.length(), index, substitute);
                } else finished = true;
            }
        }
    }

    /**
     * Searches a <code>String</code> for a specific pattern (sub-string).
     * <p/>
     * It does a pattern matching therefore. Since the brute-force
     * algorithm is much too slow for our purpose we use the well-known
     * Knuth-Morris-Pratt algorithm here for very fast searching. This
     * method is directly derived from the PXINDEX project by the same
     * author ;-) This method variation does not only give the response
     * if the pattern was found or not, but returns the next index which
     * has not been processed after the substring was found. So calling
     * methods may replace the sub-string and start searching again. If
     * the pattern was not found within the search string the method
     * returns -1.
     *
     * @param tosearch The string to be searched
     * @param pattern  This sub-string is the search target
     * @return Next unprocessed index if found, else -1
     */
    private static int searchIndex(String tosearch, String pattern) {

        //  According to Knuth-Morris-Pratt
        int i, j, M, N;
        int[] next;


        if (tosearch != null && pattern != null) {
            M = pattern.length();
            N = tosearch.length();
            next = new int[M + 1];

            if (N >= M && M > 0 && N > 0) {
                next[0] = -1;
                for (i = 0, j = -1; i < M; i++, j++, next[i] = j) {
                    while ((j >= 0) && (pattern.charAt(i) != pattern.charAt(j))) {
                        j = next[j];
                    }
                }

                for (i = 0, j = 0; j < M && i < N; i++, j++) {
                    while ((j >= 0) && (tosearch.charAt(i) != pattern.charAt(j))) {
                        j = next[j];
                    }
                }
                if (j == M) return i;
            }
        }
        return -1;
    }

    /**
     * This can be used for the application be executed standalone.
     * <p>
     * Supported arguments:
     * <ul>
     * <li><code>-jarfile [file]</code>      Optional JAR file to process, classpath is analyzed otherwise</li>
     * <li><code>-props [file]</code>        Optional configuration how to display JAR contents</li>
     * <li><code>-headless</code>            Optional flag to run application in headless mode</li>
     * </ul>
     *
     * @param args Command line arguments
     */
    public static void main(String[] args) {

        String jarFile = null;
        String props = null;
        Boolean headless = null;

        // Evaluate the command line parameters (optional, for module tests)
        for (int i = 0; i < args.length; i++) {
            if (args[i].equals("-jarfile") && (i + 1) < args.length) {
                jarFile = args[i + 1];
                i++;  // Consumed
            } else if (args[i].equals("-props") && (i + 1) < args.length) {
                props = args[i + 1];
                i++;
            } else if (args[i].equals("-headless")) {
                headless = Boolean.TRUE;
            }
        }

        // Initialization is done by the application class
        new JarInfo(jarFile, props, headless);
    }
}
